#include "global.h"

/* Functions in this file are related to the detection part 
	There are three types of functions
	1. Load invariant lists (class/object/lists)
	2. Check invariants  (Verify if invariants hold at runtime)
	3. Verification invariants invoked from Step 2 for actual checking of invariants
*/

extern FILE * inv_alerts;
extern map<string,vector<inv_ent_t> > invariants;
extern map<string,vector<inv_ent_t> > object_invariants;
extern vector<inv_ent_t> list_invariants;
extern map<string,vector<unsigned long long> > listvals;

bool verify_constant(vector<fld_ent_t> fields, string inv_str);
bool verify_set(vector<fld_ent_t> fields, string inv_str);
bool verify_bounds(vector<fld_ent_t> fields, string inv_str);

string& remove_chars(string ip_string,char srch);
string replace_with_char(string ip_string,char srch, char with);
void load_obj_invariant_list();
void load_seq_invariant_list();
char * real_trim(char * str);
char * nullify_rhs(char * str);

#define INV_BEGIN "INV_BEGIN"
#define INV_END "INV_END"
#define PPT ":::"

/* Function checks if object invariants hold */
void check_obj_invariants(string prefix, vector<fld_ent_t> fields, unsigned long long sym_addr, string pathname)
{
	//Called by the traverser after filling in data for one object
	//Look up the invariants data structure and verify all invariants hold true
	char key[MAX_LINE_INPUT_SIZE];
	char symbuf[MAX_LINE_INPUT_SIZE];
	//Key to map is in the form struct_name:::sym_addr
	sprintf(symbuf,"0x%llx",sym_addr);
	strcpy(key,prefix.c_str());
	strcat(key,PPT);
	strcat(key,symbuf);
	string strkey = key;

	map<string,vector<inv_ent_t> >::iterator inv_iter = object_invariants.find(strkey);
	//cout << "Invariants for object [" << strkey<< "] end?" << (inv_iter == object_invariants.end())<< endl;

	if(inv_iter != object_invariants.end())
	{
		//cout << "Invariants for object ::: " << strkey << endl;
		vector<inv_ent_t> current_invs = inv_iter->second;

		vector<fld_ent_t>::iterator fld_iter;
		vector<inv_ent_t>::iterator curinv_iter;

		//Verify all properties
		for(curinv_iter = current_invs.begin(); curinv_iter != current_invs.end(); curinv_iter++)
		{
			bool cannot_verify = false;
			string inv_str = (*curinv_iter).invstr;
			bool result;
			if((strstr((char *)inv_str.c_str(),"size(") != NULL) || (strstr((char *)inv_str.c_str(),"[") != NULL) && (strstr((char *)inv_str.c_str(),"]") != NULL))
			{
				//Ignore these invariants for now
				continue;
			}
			//cout << "Verifying invariant: ###" << trim((char *)inv_str.c_str()) << "###  " << endl;
			fprintf(inv_alerts,"Invariant - %s: ###%s###",strkey.c_str(), trim((char *)inv_str.c_str()));

			if((*curinv_iter).invtype == INV_CONSTANT)
			{
				result = verify_constant(fields, inv_str);
			}
			else if((*curinv_iter).invtype == INV_SET)
			{
				result = verify_set(fields, inv_str);
			}
			else if((*curinv_iter).invtype == INV_BOUNDS)
			{
				result = verify_bounds(fields, inv_str);
			}
			else
			{
				//Cannot verify invariant right now
				cannot_verify = true;

			}
			//Similarly verify all other properties here
			if(cannot_verify)
			{
				fprintf(inv_alerts,"%s\n","CANNOT VERIFY");
			}
			else
			{
				if(result == false)
				{
					fprintf(inv_alerts,"%s\n","FALSE");
					//cout << "Invariant failed for [" << pathname << "] " << strkey << ": " << inv_str << endl;
				}
				else
				{
					fprintf(inv_alerts,"%s\n","TRUE");
				}
			}
		}
	}
}

/* Function checks if list invariants hold */
void check_seq_invariants()
{
	for(int i=0; i< list_invariants.size(); i++)
	{
		inv_ent_t ent = list_invariants[i];

		if(ent.invtype == INV_SUBSET)
		{
			string invstr = ent.invstr;

			//Get names of two lists
			char buffer[MAX_LINE_INPUT_SIZE];
			strcpy(buffer,invstr.c_str());
			real_trim(buffer);

			string first = strtok(buffer," ");
			strtok(NULL," ");
			strtok(NULL," ");
			strtok(NULL," ");
			strtok(NULL," ");
			string second = strtok(NULL," ");

			map<string,vector<unsigned long long> >::iterator first_iter =  listvals.find(first);
			map<string,vector<unsigned long long> >::iterator second_iter =  listvals.find(second);
			
			if((first_iter != listvals.end()) && (second_iter != listvals.end()))
			{
				vector<unsigned long long> first_list = first_iter->second;
				vector<unsigned long long> second_list = second_iter->second;

				int num_elems_found =0;
				for(int j=0; j<first_list.size(); j++)
				{
					for(int k=0; k < second_list.size(); k++)
					{
						if(first_list[j] == second_list[k])
						{
							num_elems_found++;
							break;
						}
					}
				}
				
				if(num_elems_found == first_list.size())
				{
					//fprintf(inv_alerts,"%s\n","TRUE");
					continue;
				}
				else
				{
					fprintf(inv_alerts,"Invariant - ###%s###",trim((char *)ent.invstr.c_str()));
					fprintf(inv_alerts,"%s\n","FALSE");
				}
			}
			else
			{
				//fprintf(inv_alerts,"%s\n","LISTS NOT FOUND");
				continue;
			}
		}
		else if(ent.invtype == INV_LEN)
		{
			string invstr = ent.invstr;

			//Get names of two lists
			char buffer[MAX_LINE_INPUT_SIZE];
			strcpy(buffer,invstr.c_str());
			real_trim(buffer);
			char lenbuf[4];

			real_trim(buffer);
			char * ptr = strstr(buffer,"length ");
			strcpy(lenbuf, ptr + 7);

			string varname = strtok(buffer, " ");
			int size = atoi(lenbuf);

			map<string,vector<unsigned long long> >::iterator iter =  listvals.find(varname);
			if(iter != listvals.end())
			{
				vector<unsigned long long> var = iter->second;

				if(var.size() == size)
				{
					//fprintf(inv_alerts,"%s\n","TRUE");
					//continue;
				}
				else
				{	
					fprintf(inv_alerts,"Invariant - ###%s###",trim((char *)ent.invstr.c_str()));
					fprintf(inv_alerts,"%s\n","FALSE");
					//continue;
				}

			}
			else
			{
				//fprintf(inv_alerts,"%s\n","LIST NOT FOUND");
				//continue;
			}

			//fflush(inv_alerts);
		}
	}
}


/* Function checks class invariants */
void check_invariants(string prefix, vector<fld_ent_t> fields, string pathname)
{
	//Called by the traverser after filling in data for one object
	//Look up the invariants data structure and verify all invariants hold true
	map<string,vector<inv_ent_t> >::iterator inv_iter = invariants.find(prefix);
	//cout << "Invariants for object ::: [" << prefix << "] end?= " << (inv_iter == invariants.end())<< endl;

	if(inv_iter != invariants.end())
	{
		//cout << "Invariants for object ::: " << prefix << endl;
		vector<inv_ent_t> current_invs = inv_iter->second;

		vector<fld_ent_t>::iterator fld_iter;
		vector<inv_ent_t>::iterator curinv_iter;

		//Verify all properties
		for(curinv_iter = current_invs.begin(); curinv_iter != current_invs.end(); curinv_iter++)
		{
			bool cannot_verify = false;
			string inv_str = (*curinv_iter).invstr;
			bool result;
			if((strstr((char *)inv_str.c_str(),"size(") != NULL) || (strstr((char *)inv_str.c_str(),"[") != NULL) && (strstr((char *)inv_str.c_str(),"]") != NULL))
			{
				//Ignore these invariants for now
				continue;
			}
			//cout << "Verifying invariant: ###" << trim((char *)inv_str.c_str()) << "###  " << endl;
			fprintf(inv_alerts,"Invariant - %s: ###%s###",prefix.c_str(), trim((char *)inv_str.c_str()));

			if((*curinv_iter).invtype == INV_CONSTANT)
			{
				result = verify_constant(fields, inv_str);
			}
			else if((*curinv_iter).invtype == INV_SET)
			{
				result = verify_set(fields, inv_str);
			}
			else if((*curinv_iter).invtype == INV_BOUNDS)
			{
				result = verify_bounds(fields, inv_str);
			}
			else
			{
				//Cannot verify invariant right now
				cannot_verify = true;

			}
			//Similarly verify all other properties here
			if(cannot_verify)
			{
				fprintf(inv_alerts,"%s\n","CANNOT VERIFY");
			}
			else
			{
				if(result == false)
				{
					fprintf(inv_alerts,"%s\n","FALSE");
					//cout << "Invariant failed for [" << pathname << "] " << prefix << ": " << inv_str << endl;
				}
				else
				{
					fprintf(inv_alerts,"%s\n","TRUE");
				}
			}
		}
	}
}

/* Function checks if bounds hold for the given data structures */
// Takes care of != , > , < , >= , <=
bool verify_bounds(vector<fld_ent_t> fields, string inv_str)
{
	vector<fld_ent_t>::iterator iter;
	char name_buffer[MAX_LINE_INPUT_SIZE];
	char value_buffer[MAX_LINE_INPUT_SIZE];
	string field_name;
	string field_value;
	string alt_field_value = "0";
	char * ptr;
	string op;

	//Extract relevant tokens here
	strcpy(name_buffer,inv_str.c_str());
	strcpy(value_buffer,inv_str.c_str());
	real_trim(name_buffer);
	real_trim(value_buffer);
	if((ptr = strstr(name_buffer, " ")) != NULL)
	{
		ptr[0] ='\0';	
		field_name.append(name_buffer);
	}

	ptr = strstr(name_buffer, " ");

	if((ptr = strstr(value_buffer, "!=")) != NULL)
	{
		field_value.append(ptr + 3);
		op = "!=";	
	}
	else if((ptr = strstr(value_buffer, "<=")) != NULL)
	{
		field_value.append(ptr + 3);
		op = "<=";	
	}
	else if((ptr = strstr(value_buffer, "<")) != NULL)
	{
		field_value.append(ptr + 3);
		op = "<";	
	}
	else if((ptr = strstr(value_buffer, ">=")) != NULL)
	{
		field_value.append(ptr + 3);
		op = ">=";	
	}
	else if((ptr = strstr(value_buffer, ">")) != NULL)
	{
		field_value.append(ptr + 3);
		op = ">";	
	}

	//cout << "Field name = " << field_name << " field value=" << field_value << " op =" << op << endl;
	
	for(iter = fields.begin(); iter != fields.end(); ++iter)
	{
		fld_ent_t flds = *iter;

		if(field_name == flds.field_name)
		{
			if(flds.isstrfield)
			{
				if(op == "!=")
				{
					if(field_value != flds.str_field_value)
					{
						//fprintf(inv_alerts,"[%s]\n",field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == "<")
				{
					if(flds.str_field_value < field_value)
					{
						//fprintf(inv_alerts,"[%s]\n",field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == "<=")
				{
					if(flds.str_field_value <= field_value)
					{
						//fprintf(inv_alerts,"[%s]\n",field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == ">")
				{
					if(flds.str_field_value > field_value)
					{
						//fprintf(inv_alerts,"[%s]\n",field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == ">=")
				{
					if(flds.str_field_value >= field_value)
					{
						//fprintf(inv_alerts,"[%s]\n",field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
			}
			else
			{
				unsigned long long v1,v2;
				v1 = strtoll(flds.numeric_field_value.c_str(), NULL,10);
				v2 = strtoll(field_value.c_str(), NULL,10);
				if(op == "!=")
				{
					//if(field_value != flds.numeric_field_value)
					if (v1 != v2)
					{
						//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == "<")
				{
					//if(flds.numeric_field_value < field_value)
					if (v1 < v2)
					{
						//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == "<=")
				{
					//if(flds.numeric_field_value <= field_value)
					if (v1 <= v2)
					{
						//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == ">")
				{
					//if(flds.numeric_field_value > field_value)
					if (v1 > v2)
					{
						//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
				else if(op == ">=")
				{

					//if(flds.numeric_field_value >= field_value)
					if (v1 >= v2)
					{
						//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
						return(true);
					}
					else
					{
						//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
						return(false);
					}
				}
			}
		}
	}
	return(false);
}

/* Function verifies if set invariant holds for the given data structures */
bool verify_set(vector<fld_ent_t> fields, string inv_str)
{
	vector<fld_ent_t>::iterator iter;
	char name_buffer[MAX_LINE_INPUT_SIZE];
	char value_buffer[MAX_LINE_INPUT_SIZE];

	//Extract relevant tokens here
	strcpy(name_buffer,inv_str.c_str());
	strcpy(value_buffer,inv_str.c_str());
	trim(name_buffer);
	trim(value_buffer);
	string field_name;
	string field_value;
	string alt_field_value = "0";

	char * ptr;
	ptr = strstr(name_buffer, " ");
	if((ptr = strstr(name_buffer, " ")) != NULL)
	{
		ptr[0] ='\0';	
		field_name.append(name_buffer);
	}
	if((ptr = strstr(value_buffer, "one of")) != NULL)
	{
		field_value.append(ptr + 7);
	}
	//fprintf(inv_alerts," Expecting field value [%s]\n ",field_value.c_str());

	//Toeknize string and extract separate values
	char buffer[MAX_LINE_INPUT_SIZE];
	vector<string> vals;
	strcpy(buffer,field_value.c_str());

	ptr = strtok(buffer,",{");
	while(ptr != NULL)
	{
		vals.push_back(ptr);
		ptr = strtok(NULL, ",}");
	}

	//Values in the vector contain leading and trailing space and double quotes
	for(iter = fields.begin(); iter != fields.end(); ++iter)
	{
		fld_ent_t flds = *iter;

		if(field_name == flds.field_name)
		{
			//fprintf(inv_alerts," Comparing field [%s] = ",field_name.c_str());
			vector<string>::iterator sit = vals.begin();
	
			while(sit != vals.end())
			{
				field_value = *sit;
				replace_with_char(field_value,'"',' ');
				field_value = real_trim((char *)field_value.c_str());
				if(flds.isstrfield)
				{
					//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.str_field_value.c_str(), field_value.c_str());
					if(field_value == flds.str_field_value)
					{
						//fprintf(inv_alerts,"[%s] ",field_value.c_str());
						return(true);
					}
				}
				else
				{
					//fprintf(inv_alerts," fields value [%s] [%s]\n ",flds.numeric_field_value.c_str(), field_value.c_str());
					if(field_value == "null")
						field_value = "0";

					unsigned long long v1,v2;
					v1 = strtoll(field_value.c_str(), NULL,10);
					v2 = strtoll(flds.numeric_field_value.c_str(), NULL,10);
						//fprintf(inv_alerts,"[%lld %lld]\n",v1,v2);
					if(v1 == v2)
					{
						// fprintf(inv_alerts,"[%s] ",field_value.c_str());
						return(true);
					}
				}
				++sit;
			}
		}
	}
	return(false);
}

/* Function verifies if the given data structure value is a constant */
bool verify_constant(vector<fld_ent_t> fields, string inv_str)
{
	vector<fld_ent_t>::iterator iter;
	char name_buffer[MAX_LINE_INPUT_SIZE];
	char value_buffer[MAX_LINE_INPUT_SIZE];
	//Extract relevant tokens here

	strcpy(name_buffer,inv_str.c_str());
	strcpy(value_buffer,inv_str.c_str());
	trim(name_buffer);
	trim(value_buffer);
	string field_name;
	string field_value;
	string alt_field_value = "0";

	char * ptr;
	ptr = strstr(name_buffer, " ");
	if((ptr = strstr(name_buffer, " ")) != NULL)
	{
		ptr[0] ='\0';	
		field_name.append(name_buffer);
	}
	if((ptr = strstr(value_buffer, "has only one value ")) != NULL)
	{
		field_value.append(ptr + 19);
	}
	else if((ptr = strstr(value_buffer, "==")) != NULL)
	{
		field_value.append(ptr + 3);
	}
	
	for(iter = fields.begin(); iter != fields.end(); ++iter)
	{
		fld_ent_t flds = *iter;

		if(field_name == flds.field_name)
		{
			if(flds.isstrfield)
			{
				if(field_value == flds.str_field_value)
				{
					//fprintf(inv_alerts,"[%s] ",field_value.c_str());
					return(true);
				}
				else
				{
					//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
					return(false);
				}
			}
			else
			{
				unsigned long long v1,v2;
				v1 = strtoll(flds.numeric_field_value.c_str(), NULL,10);
				v2 = strtoll(field_value.c_str(), NULL,10);
				
				//if(field_value == flds.numeric_field_value)
				if(v1 == v2)
				{
					return(true);
				}
				else
				{
					//fprintf(inv_alerts," Unmatched value = [%s]\n ",flds.numeric_field_value.c_str());
					return(false);
				}
			}
		}
	}
	return(false);
}


/* Assign an invariant type to the given string */
inv_types identify_invariant_type(string inv)
{
	inv_types ret = INV_UNKNOWN;
	//Parse string and identify the invariant type - return type

	if(strstr(inv.c_str(),"size(") != NULL)
		return(INV_LEN);
	if(strstr(inv.c_str(),"length") != NULL)
		return(INV_LEN);
	else if(strstr(inv.c_str(),"has only one value") != NULL)
		return(INV_CONSTANT);
	else if(strstr(inv.c_str()," == ") != NULL)
		return(INV_CONSTANT);
	else if(strstr(inv.c_str()," != ") != NULL)
		return(INV_BOUNDS);
	else if(strstr(inv.c_str()," > ") != NULL)
		return(INV_BOUNDS);
	else if(strstr(inv.c_str()," < ") != NULL)
		return(INV_BOUNDS);
	else if(strstr(inv.c_str()," >= ") != NULL)
		return(INV_BOUNDS);
	else if(strstr(inv.c_str()," <= ") != NULL)
		return(INV_BOUNDS);
	else if(strstr(inv.c_str()," one of ") != NULL)
		return(INV_SET);
	else if(strstr(inv.c_str()," is a subset of ") != NULL)
		return(INV_SUBSET);
		
	return(ret);
}

/* Function loads list invariants */
#define SKIP_LINE "====================="
void load_seq_invariant_list()
{
	//Invariant for each type is precedes with line SKIP_LINE and succeeds with line SKIP_LINE 
	cout << "Loading list invariants" << endl;
	FILE * fd_in;
	
	fd_in = fopen("invariant-list/ll.length","r");
	if(fd_in == NULL)
	{
		cout << "Cannot open file invariant-list/list.*" << endl;
		exit(-1);
	}

	char buffer[MAX_LINE_INPUT_SIZE];
	int inv_count =0;

	fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	while(!feof(fd_in))
	{
		if((strstr(buffer, SKIP_LINE) != NULL) || (strstr(buffer,PPT) != NULL))
		{
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
			continue;
		}
		else
		{
			//New struct encountered
			inv_ent_t ent;
			//Add the invariant to invlist

			char * ptr = strstr(buffer,".");
			while(ptr != NULL)
			{
			if(ptr != NULL)
			{
				//Replace . with ->
				ptr[0] = '-';
				char tmpbuf[MAX_LINE_INPUT_SIZE];
				strcpy(tmpbuf, ptr + 1);
				ptr++;
				ptr[0]='>';
				ptr++;
				ptr[0]='\0';

				strcat(buffer,tmpbuf);
			}
			ptr = strstr(buffer,".");
			}
			ent.invstr = buffer;
			ent.invtype = identify_invariant_type(ent.invstr);
			list_invariants.push_back(ent);
			++inv_count;
		}
		fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	}
	fclose(fd_in);
	cout << "Total # of list invariants loaded = " << inv_count << endl;
}


/* Function loads the generalized object invariant list - ignore these, found to be useless */
void load_generalized_obj_invariant_list()
{
	/* This function assumes that load_obj_invariant_list() has already created the map */
	map<string,vector<inv_ent_t> >::iterator iter = object_invariants.begin();

	string key = iter->first;
	string defname = strtok((char *)key.c_str(),PPT);
	string refdefname = defname;
	vector<inv_ent_t> refinvlist = iter->second;	

	FILE * fdout = fopen("invariant-list/obj.constancy.generic","w");
	if(fdout == NULL)
	{
		cout << "Cannot open file invariant-list/obj.constancy.generic" << endl;
		exit(-1);
	}

	cout << "Creating generalized object invariants " << refinvlist.size() <<  endl;
	for(int i=0; i< refinvlist.size(); i++)
	{
		if(refinvlist[i].invtype == INV_CONSTANT)
		{
			refinvlist[i].invstr = nullify_rhs((char *)refinvlist[i].invstr.c_str());
		}
	}

	while(iter != object_invariants.end())
	{
		vector<inv_ent_t> invlist;	
		key = iter->first;
		defname = strtok((char *)key.c_str(),PPT);
		invlist = iter->second;

		if(defname == refdefname)
		{
			vector<inv_ent_t>::iterator viter = refinvlist.begin();

			for(int i=0; i< refinvlist.size(); i++)
    		{
				bool exists = false;
        		if(refinvlist[i].invtype == INV_CONSTANT)
        		{
					for(int j=0; j < invlist.size(); j++)
					{
						if(invlist[j].invtype == INV_CONSTANT)
						{
							if(strstr(invlist[j].invstr.c_str(), refinvlist[i].invstr.c_str()) != NULL)
							{
								exists = true;
								break;
							}
						}
					}

					if(!exists)
					{
						refinvlist.erase(viter);	
					}
				}
				++viter;
			}

		}
		else
		{

			fprintf(fdout, ":::%s\n", refdefname.c_str());

			for(int j=0; j < refinvlist.size(); j++)
			{
				if(refinvlist[j].invtype == INV_CONSTANT)
				{
					fprintf(fdout, "%s\n", refinvlist[j].invstr.c_str());
				}
			}

			refdefname = defname;
			refinvlist = invlist;	
			for(int i=0; i< refinvlist.size(); i++)
    		{
        		if(refinvlist[i].invtype == INV_CONSTANT)
        		{
            		refinvlist[i].invstr = nullify_rhs((char *)refinvlist[i].invstr.c_str());
        		}
    		}
		}
		
		++iter;
	}
	fclose(fdout);
}

char * nullify_rhs(char * str)
{
	char * ptr = strstr(str,"==");
	if (ptr != NULL)
	{
		ptr[3]='\0';
	}
	else
	{
		ptr = strstr(str,"has only one value ");
		ptr[19] = '\0';
	}

	return(str);
}


/* Function loads the object invariant list */
void load_obj_invariant_list()
{
	//Invariant for each type is precedes with line INV_BEGIN and succeeds with line INV_END
	//May objects of same type are separated with SKIP_LINE
	FILE * fd_in;
	
	fd_in = fopen("invariant-list/obj.bounds","r");

	if(fd_in == NULL)
	{
		cout << "Cannot open file invariant-list/obj.*" << endl;
		exit(-1);
	}

	char buffer[MAX_LINE_INPUT_SIZE];
	string defname;
	vector<inv_ent_t> invlist;
	int inv_per_count;
	int inv_count =0;

	fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	while(!feof(fd_in))
	{
		if(strstr(buffer,INV_BEGIN) != NULL)		
		{
			//New struct encountered
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
			if(strstr(buffer, SKIP_LINE) != NULL)
				fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);

			if(strcmp(trim(buffer),INV_END) == 0)
			{
				fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
				continue;
			}
			
			inv_per_count=0;

			//defname = strtok(buffer,PPT);
			defname = trim(buffer);
			//cout << "defname=[" << defname << "]" << endl;
			assert(defname.size() > 0);
		}
		else if(strstr(buffer, SKIP_LINE) != NULL)
		{
			//Insert old one
			if(inv_per_count > 0)
			{
				//Insert item into map
				object_invariants.insert(make_pair(defname,invlist));
			}
			//Clear so that new one can be started
			invlist.clear();
			defname.clear();
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);

			if(strstr(buffer, INV_END) != NULL)
			{
				fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
				continue;
			}
			else
			{
				//Get new one
				inv_per_count=0;
            	//defname = strtok(buffer,PPT);
				defname = trim(buffer);
            	assert(defname.size() > 0);
			}

		}
		else if(strstr(buffer,INV_END) != NULL)
		{
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
			if(invlist.size() > 0)
			{
				//Insert item into map
				object_invariants.insert(make_pair(defname,invlist));
				invlist.clear();
				defname.clear();
			}
			continue;
		}
		else
		{
			inv_ent_t ent;
			//Add the invariant to invlist
			ent.invstr = buffer;
			ent.invtype = identify_invariant_type(ent.invstr);
			invlist.push_back(ent);
			//cout << ent.invstr << endl;
			++inv_per_count;
			++inv_count;
		}
		fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	}
	fclose(fd_in);
	cout << "Total # of invariants loaded = " << inv_count << endl;
	cout << "Invariants loaded for " << object_invariants.size() << " objects" << endl;
}

/* Function loads the class invariant list */
void load_invariant_list()
{
	//Invariant for each type is precedes with line INV_BEGIN and succeeds with line INV_END
	FILE * fd_in;
	
	fd_in = fopen("invariant-list/class.constancy","r");
	if(fd_in == NULL)
	{
		cout << "Cannot open file invariant-list/class.*" << endl;
		exit(-1);
	}

	char buffer[MAX_LINE_INPUT_SIZE];
	string defname;
	vector<inv_ent_t> invlist;
	int inv_per_count;
	int inv_count =0;

	fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	while(!feof(fd_in))
	{
		if(strstr(buffer, SKIP_LINE) != NULL)
		{
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
			continue;
		}
		if(strstr(buffer,INV_BEGIN) != NULL)		
		{
			//New struct encountered
			fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
			if(strstr(buffer, SKIP_LINE) != NULL)
				fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);

			assert(strcmp(buffer,INV_END) != 0);
			inv_per_count=0;

			defname = strtok(buffer,PPT);
			assert(defname.size() > 0);
		}
		else if(strstr(buffer,INV_END) != NULL)
		{
			if(inv_per_count > 0)
			{
				//Insert item into map
				invariants.insert(make_pair(defname,invlist));
			}
			//Clear so that new one can be started
			invlist.clear();
			defname.clear();
		}
		else
		{
			inv_ent_t ent;
			//Add the invariant to invlist
			ent.invstr = buffer;
			ent.invtype = identify_invariant_type(ent.invstr);
			invlist.push_back(ent);
			++inv_per_count;
			++inv_count;
		}
		fgets(buffer,MAX_LINE_INPUT_SIZE,fd_in);
	}
	fclose(fd_in);
	cout << "Total # of invariants loaded = " << inv_count << endl;
	cout << "Invariants loaded for " << invariants.size() << " classes" << endl;
}
