/*
 * Author: Arati Baliga 
 * Project: Gibraltar v 1.0
 *
 * Goal: To deduce set based invariants in the kernel static and dynamic areas 
 *       using a Daikon-like approach
 *
 * 		 # Monitoring code collects data from remote machine memory in blocks
 * 		 # Makes sequential passes over the static area and uses garbage collector type
 * 		   approach to collect dynamic objects 
 * 
*/
#include "monitor.h"
#include "sha1.c"

#define PRINT_DAIKON_TRACES 1

#define VERBOSE 2  
/* VERBOSE =0 , Don't display any messages 
   VERBOSE =1 , Display few messages
   VERBOSE =2 , Display all messages, good for debugging
*/
#define PRINT_TO_FILE 1
#define LOG 1
#define INV_LOG 1

//#define WHITE_LIST 1

/* Variables below are used by the Myrinet page fetching code */
//TODO CHANGE
/*static struct gm_port *p = NULL;
static void  *in_buffer = NULL, *in_buffer_sec=NULL;
int dest_node_id;
gm_recv_event_t *e;
char * out_buffer = NULL;
*/

FILE * fd_filter;	
queue<entry_t> listq; //Stores nodes that are heads of a linked list

void my_handler(int s) {
	printf("caught signal %d\n",s);
	printf("total_checks: %d\n",total_checks);
	printf("unique structs: %d\n",unique_structs.size());
	printf("avg checks: %d\n",total_checks/unique_structs.size());
	gettimeofday(&end,NULL);
	printf("Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));
	exit(1);
}

/* Signal handler for Ctrl-C. Unused right now */
short ctrlc_counter =0;
void  INThandler(int sig)
{/*
    signal(sig, SIG_IGN);
	ctrlc_counter++;

	if(ctrlc_counter == 1)
	{
		 //print_invariants("invariants.txt",total_page_frames);
         detection = 1;
         printf("Switched to detection phase\n");
	}
	else
	{
    	printf("Hit Ctrl-C .. Quitting!\n");
		exit_now = 1;
	
		if(detection == 0)
		{
			printf("Learning stage did not end yet. Will not produce correct results\n");
		}
		//exit_code();
	}*/
    signal(SIGINT, INThandler); 
}

/* Prefetches all memory pages from the target at the beginning of the measurement round */
void prefetch_pages()
{
    unsigned long long all_start = 0xc0101000;
    //unsigned long long all_start= 0xc0285000; //begin of static area
    //unsigned long long all_start = 0xc032d000; //end of static area
    unsigned long long all_end= 0xf7FFF000;
                                                                                                                                               
    prefetch_phase = true;
    cout << "Prefetching Pages - start" << get_time() << endl;
    while(all_start < all_end)
    {
		if(valid_page(all_start))
		{
        	get_page(all_start,1);
		}
        all_start += SLAB_SIZE;
    }
    prefetch_phase = false;
    cout << "Prefetching pages - done" << get_time() << endl;
                                                                                                                                               
}


/* Generates a list of invariants in static data */
void generate_static_invariant_list()
{
	FILE * fd;
	struct stat s;

    if(stat("input/static.invariants",&s)  == -1)
    {
		//Only if file does not exist, generate a fresh list
		fd = fopen("input/static.invariants","w");
		if(fd == NULL)
		{
			cout << "Cannot open file input/static.invariants" << endl;
			exit(-1);
		}

		for(int i=0; i<static_symbol.size(); i++)
		{
			if(static_symbol[i].invariant)
			{
				fprintf(fd, "%llx %s\n", static_symbol[i].sym_addr, static_symbol[i].sym_name);
			}
		}
		fclose(fd);
	}

}


int main(int argc, char **argv)
{
	static int recv_sz;
	char dest_node_name[64];
	unsigned int my_node, global_id;
	int ret,i,send_tokens;
	int block_no = 0;
	unsigned int block_size;
	int msmt_round = 1;
	int mem_start, mem_end,current_mem_start, chunk_size;
	int err;
	SHA1Context sha;
	char page_filename[FILENAME_SIZE];
	int fd_terminate;
	FILE * fd_staticmap;
	int optimization;

	//Check number of arguments
	if ((argc != 8)){
		printf("USAGE: recv <domain id> <mem start> <mem end> <msmt_rounds> <mode> <optimization> <variable>\n");
		exit(-1);
	}

	//Activate the Ctrl-C signal handler
//   	signal(SIGINT, INThandler);
	signal(SIGINT,my_handler);
	
	dom = atoi(argv[1]);
	xainit = 0;
	mem_start = strtol(argv[2],NULL,16);
	chunk_size = (SLAB_SIZE/2); 
	current_mem_start = mem_start;
	mem_end = strtol(argv[3], NULL,16); 
	MSMT_ROUNDS = atoi(argv[4]);
	total_page_frames = ((mem_end - mem_start) / SLAB_SIZE);
	detection =atoi(argv[5]);
	optimization = atoi(argv[6]);	

	if(optimization == 0)
	{
		shadow = 0;
		notify = 0;
	}
	else if(optimization == 1)
	{
		shadow = 1;
		notify = 0;
		time_between = atoi(argv[7]);	
	}
	else
	{
		shadow = 1;
		notify = 1;
		num_pages = atoi(argv[7]);
	}

	total_checks = 0;

	//Mode: 0 - Training mode, 1 - Detection mode
	cout << "Mode =" << detection << endl;
	printf("Msmt rounds = %d\n", MSMT_ROUNDS);
	printf("Total page frames to monitor = %d\n", total_page_frames);

	//Sanity checks
	if((mem_start <= 0xa0000) && (mem_end >= 0xBffff))
	{
		printf("The address range you have specified contains the PCI device space from 0xA0000 - 0xBFFFF.\n " \
		       "Please exclude the range.\n");
		exit(-1);
	}
	if(((mem_start - mem_end) % SLAB_SIZE) != 0)
	{
		printf("Address range is not an exact multiple of the block size.\n");
		exit(-1);
	}
	
	/* Initialize XenAccess API */
	
	printf("initialize XenAccess library\n");
	if (xa_init_vm_id_strict(dom, &xai) == XA_FAILURE){
		perror("failed to init XenAccess library\n");
		exit(-1);
	}
	
	live_p2m = NULL;
	live_m2p = NULL;

	int xc_handle = xc_interface_open();
	
	if(xc_handle == -1)
	{
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}

	memory_map = build_memory_map(dom);
	live_p2m = get_p2m_table(xc_handle, dom);
	live_m2p = get_m2p_table(xc_handle, dom);

	p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &dom) + 1;
	
	translate = 0;

	printf("Mem start = 0x%x\n", mem_start);
	printf("Mem end= 0x%x\n", mem_end);
	printf("Block size= 0x%x %d\n", SLAB_SIZE, SLAB_SIZE);

	gettimeofday(&start, NULL);

	/*Read the static memory map */
	fd_staticmap = fopen("input/memory.map.static","r");
	if(fd_staticmap == NULL)
	{
		printf("Cannot open file input/memory.map.static\n");
		exit(-1);
	}

	//Read static symbols in memory before beginning fetch
	cout << "[" << get_time() << "] Begin - Reading static map" << endl;
	read_static_map();
	cout << "[" << get_time() << "] End - Reading static map" << endl;
	cout << "[" << get_time() << "] Begin - Populating initial set of roots" << endl;
	populate_start_set();
	cout << "[" << get_time() << "] End - Populating initial set of roots" << endl;

	//Build map of type definitions , offsets and field sizes
	cout << "[" << get_time() << "] Begin - Reading type definitions" << endl;
	build_definition_map();	
	cout << "[" << get_time() << "] End - Reading type definitions" << endl;
	//print_definition_map();
    fd_stats = fopen("stats.txt","w");
    if(fd_stats == NULL)
    {
        printf("Cannot open file stats.txt\n");
        exit(-1);
    }
	fprintf(fd_stats,"Tot_Objs Tot_Syms ObjswFp Tot_Fps Time\n");

    fd_alerts = fopen("alerts.txt","w");
    if(fd_alerts == NULL)
    {
        printf("Cannot open file alerts.txt\n");
        exit(-1);
    }
    inv_alerts = fopen("inv_alerts.txt","w");
    if(inv_alerts == NULL)
    {
        printf("Cannot open file inv_alerts.txt\n");
        exit(-1);
    }

#ifdef INV_LOG
	inv_log = fopen("log/inv_log.txt","w");
	if(inv_log == NULL)
	{
		printf("Cannot open log file log/inv_log.txt\n");
		exit(-1);
	}
	fprintf(inv_log,"Invariant Log Initialized\n");
	fflush(inv_log);
#endif


    //fprintf(fd_stats,"Root set size =%d\n",init_roots.size());
    //fprintf(fd_stats,"Null roots = %d\n\n",null_roots);

	bool invlist_loaded = false;
	
	/* JEFF added bounds to detection phase */
	while(((msmt_round < MSMT_ROUNDS) && (detection ==0)) || ((msmt_round <= MSMT_ROUNDS) && detection == 1))
	{
#ifdef LOG
		log_init();
#endif
#ifdef INV_LOG
			fprintf(inv_log,"Begin Round %d\n",msmt_round);		
			fflush(inv_log);	
#endif	

		char logbuffer[MAX_LINE_INPUT_SIZE];
		gettimeofday(&scan_start, NULL);
        	printf("Mode = %d, Msmt pass =%d\n", detection,msmt_round);

		//prefetch_pages();
		cout << "page map size = "<< page_map.size() << endl;
		if((!invlist_loaded) && (detection == 1))
		{
			cout << "[" << get_time() << "] Begin - Loading invariant list" << endl;
			//load_invariant_list(); // Activate as required - loads class invariants
			load_obj_invariant_list(); // Activate as required - loads object invariants

			/* Goal was to extract generic properties across similar objects - Found to be quite useless
				as it was severely limiting the number of invariants */	
			//load_generalized_obj_invariant_list(); //- Ignore this

			//load_seq_invariant_list(); // Activate as required, loads list invariants
			cout << "[" << get_time() << "] End - Loading invariant list" << endl;
			invlist_loaded = true;
		}

		total_objects =0;
		total_symbols = 0;
		null_roots = 0;
		obj_func_ptr =0;
		total_func_ptr =0;
		inconsistent_cnt = 0;
		same_object_cnt = 0;

			//Used for debugging to look at dangling pointers
			//-----------
		fd_dangle = fopen("dangling.txt","w");
		if(fd_dangle == NULL)
		{
			printf("Cannot open file dangling.txt\n");
			exit(-1);
		}
			//-----------

		cout << "[" << get_time() << "] Read static invariant list" << endl;
		//Read static invariant list and set appropriate data structures
		if((detection == 1) && (msmt_round == 1))
		{
			FILE * fd;
			fd = fopen("input/static.invariants","r");

			if(fd == NULL)
			{
				cout << "Cannot open file input/static.invariants" << endl;
				exit(-1);
			}

			char buffer[MAX_LINE_INPUT_SIZE];
			fgets(buffer, MAX_LINE_INPUT_SIZE,fd);
			int current = 0;
			while(!feof(fd))
			{
				//printf("buffer: %s",buffer);
				char * ptr = strtok(buffer, " ");
				
				for(int i=current; i<static_symbol.size(); i++)				
				{
					static_symbol[i].invariant = false;
					static_symbol[i].alert = false;
					static_symbol[i].reported = false;
					
					if(static_symbol[i].sym_addr == strtoll(ptr,NULL,16))	
					{
						current = i + 1;		
						static_symbol[i].invariant = true;
						break;
					}
				}

				fgets(buffer, MAX_LINE_INPUT_SIZE,fd);
			}
			fclose(fd);
		}
		cout << "[" << get_time() << "] Done reading static invariant list" << endl;		

		if(shadow == 0)
		{
#ifdef INV_LOG
			fprintf(inv_log,"Normal Scan\n");		
			fflush(inv_log);	
#endif			
			fprintf(stderr,"start static scan\n");
	
			scan_static_memory(mem_start, mem_end,msmt_round);
			fprintf(stderr, "end static scan start dynamic scan\n");
			
			scan_dynamic_memory(msmt_round);
			fprintf(stderr, "end dynamic scan\n");
		}
		else
		{

#ifdef INV_LOG
			fprintf(inv_log,"Shadow Scan\n");
			fprintf(inv_log,"Start Initial Static Scan\n");
			fflush(inv_log);		
#endif
			fprintf(stderr,"start static scan\n");
			scan_static_memory(mem_start, mem_end,msmt_round);
#ifdef INV_LOG
			fprintf(inv_log,"End Initial Static Scan\n");
			fprintf(inv_log,"Start Initial Dynamic Scan\n");	
			fflush(inv_log);		
#endif
			fprintf(stderr, "end static scan start dynamic scan\n");
			scan_dynamic_memory(msmt_round);
#ifdef INV_LOG
			fprintf(inv_log,"End Initial Dynamic Scan\n");	
			fflush(inv_log);		
#endif
			visited.clear();
			page_map.clear();
			fprintf(stderr, "end dynamic scan start shadow scan\n");
			printf("dynamic_pfn_map size: %d\n",dynamic_pfn_map.size());

			msmt_round++;

			shadow_scan();

			gettimeofday(&end, NULL);
			printf("Completed measurement rounds = %d\n",20);
			printf("Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));
			
			fclose(inv_alerts);
			fclose(fd_stats);
			fclose(fd_alerts);
			gettimeofday(&end, NULL);
		
			xa_destroy(&xai);
			return 0;
		}
		
		//Clear cached page map
		page_map.clear();
		//Clear the visited map
		visited.clear();
																													
		//Clear static data
		//clear_static_data();
		gettimeofday(&scan_end, NULL);
		print_stats(msmt_round,(scan_end.tv_sec - scan_start.tv_sec));
		
		msmt_round++;
		translate = 0;

		if((msmt_round == MSMT_ROUNDS) && (detection ==0)) 
		{
			cout << "Finished training ..switching to detection mode" << endl;
			detection = 1;

			//Generate a file with data containing all static symbols that are invariant
			generate_static_invariant_list();
		}

		fclose(fd_dangle);
#ifdef LOG
		log_finish();
#endif
		fprintf(stderr, "done round\n");
	}
	
	gettimeofday(&end, NULL);
	printf("Completed measurement rounds = %d\n", (msmt_round - 1));
	printf("Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));

#ifdef INV_LOG
	fprintf(inv_log,"Completed measurement rounds = %d\n", (msmt_round - 1));
	fprintf(inv_log,"Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));	
	fflush(inv_log);		
#endif

	if(detection == 1)
	{
		fclose(fd_alerts);
	}

	fclose(inv_alerts);
	fclose(fd_stats);

	fclose(inv_log);
	munmap(memory_map,p2m_size * PAGE_SIZE);
	munmap(live_p2m,p2m_size * sizeof(xen_pfn_t));
	munmap(live_m2p,p2m_size * sizeof(xen_pfn_t));

	/* cleanup any memory associated with the XenAccess instance */
	xa_destroy(&xai);

/* TODO CHANGE
	gm_dma_free(p,out_buffer);
	gm_dma_free(p, in_buffer);
	gm_close(p);

	gm_finalize();
*/
   	return(0);
}

/* JEFF ADDITIONS */
/* needed for shadow_scan() ADD TO SEPERATE FILES LATER */
static inline int test_bit (int nr, volatile void * addr)
{
    return (BITMAP_ENTRY(nr, addr) >> BITMAP_SHIFT(nr)) & 1;
}

void *xg_memalign(size_t alignment, size_t size)
{
#if defined(_POSIX_C_SOURCE) && !defined(__sun__)
    int ret;
    void *ptr;
    ret = posix_memalign(&ptr, alignment, size);
    if (ret != 0)
        return NULL;
    return ptr;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
    return valloc(size);
#else
    return memalign(alignment, size);
#endif
}

void shadow_scan()
{
	fprintf(stderr,"in shadow scan\n");
	printf("in shadow scan\n");
	int frc, xc_handle;
	xc_shadow_op_stats_t stats;
	bitmap = NULL;
	int shadow_round = 2;
	int static_pages;

	xc_handle = xc_interface_open();
	
	if(xc_handle == -1)
	{
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}

	/* Live suspend. Enable log-dirty mode. */
	if ( xc_shadow_control(xc_handle, dom,
						XEN_DOMCTL_SHADOW_OP_ENABLE_LOGDIRTY,
						NULL, 0, NULL, 0, NULL) < 0 )
	{
		/* log-dirty already enabled? There's no test op,
			so attempt to disable then reenable it */
		frc = xc_shadow_control(xc_handle, dom, XEN_DOMCTL_SHADOW_OP_OFF,
							NULL, 0, NULL, 0, NULL);
		if ( frc >= 0 )
		{
			frc = xc_shadow_control(xc_handle, dom,
								XEN_DOMCTL_SHADOW_OP_ENABLE_LOGDIRTY,
								NULL, 0, NULL, 0, NULL);
		}
		
		if ( frc < 0 )
		{
			fprintf(stderr,"Couldn't enable shadow mode (rc %d) (errno %d)", frc, errno );
			exit(-1);
		}
	}

	/* if notify is 0 - check dirty bits every interval of time   */
    /* if notify is 1 - recieve notifications from the hypervisor */
	if(notify == 0)
	{
#ifdef INV_LOG
		fprintf(inv_log,"Shadow Scan - Fixed Interval\n");	
		fflush(inv_log);		
#endif
		bitmap = (unsigned long *)xg_memalign(PAGE_SIZE, ROUNDUP(BITMAP_SIZE, PAGE_SHIFT)); 

		if ( !bitmap )
		{
			fprintf(stderr,"Couldn't allocate bitmap array");
			exit(-1);
		}
	
		memset(bitmap, 0xff, BITMAP_SIZE);


		while(1)//(shadow_round <= MSMT_ROUNDS)
		{

			struct timeval round_start,round_end;
			gettimeofday(&round_start,NULL);

#ifdef INV_LOG
			fprintf(inv_log,"[%s] Start Shadow Round %d.\n",get_time(),shadow_round);
			fflush(inv_log);			
#endif
			static_count = 0;
			static_pages = 0;

			fprintf(stderr,"start shadow round\n");
			int i;

			if ( xc_shadow_control(xc_handle, dom, 
								XEN_DOMCTL_SHADOW_OP_CLEAN, bitmap, 
								p2m_size, NULL, 0, &stats) != p2m_size )
			{
				fprintf(stderr,"Error flushing shadow PT");
				exit(-1);
			}

			fprintf(stderr,"stats: dirty: %d\n",stats.dirty_count);

#ifdef INV_LOG
			fprintf(inv_log,"\nNumber of Dirty Pages: %d\n\n",stats.dirty_count);
			fflush(inv_log);			
#endif

			cout << "Scan p2m for dirty page - start " << get_time() << endl;
			for(i = 0; i < p2m_size; i++)
			{
				/* check for dirty page */
				if(test_bit(i,bitmap) == 1)
				{
					unsigned int addr = i << PAGE_SHIFT;

					STATIC_SYMBOL_CURRENT = -1;

					multimap<unsigned long long, int>::iterator iter = static_symbol_map.find(i);
					if(iter != static_symbol_map.end())
					{
						STATIC_SYMBOL_CURRENT = iter->second;
					}

					if(STATIC_SYMBOL_CURRENT != -1)
					{
						static_pages++;
						fetch_page(addr,0,shadow_round);
					//	printf("save_data_values STATIC_SYMBOL_CURRENT: %d\n",STATIC_SYMBOL_CURRENT);
						save_data_values(addr, shadow_round);
					}
					else
					{	
						multimap<unsigned long long, entry_t>::iterator iter = dynamic_pfn_map.find(i);
						if(iter != dynamic_pfn_map.end())
						{
							//printf("found changed dynamic symbol name: %s  type: %s\n",iter->second.pathname.c_str(), iter->second.sym_type);
					
							pair<multimap<unsigned long long,entry_t>::iterator,multimap<unsigned long long,entry_t>::iterator> entries;
							entries = dynamic_pfn_map.equal_range(i);
							int entrycount = 0;
							multimap<unsigned long long, entry_t>::iterator itr;

							for (itr=entries.first; itr!=entries.second; ++itr)
							{
								dirty_roots.push(itr->second);
								entrycount++;
							}
						}
					}
				}
			}

#ifdef INV_LOG
			fprintf(inv_log,"\nNumber of Dirty Roots: %d\n",dirty_roots.size());
			fprintf(inv_log,"Number of Static Symbols: %d in %d pages\n\n",static_count, static_pages);
			fflush(inv_log);			
#endif
			fprintf(stderr,"dirty roots: %d\n",dirty_roots.size());	
			scan_dynamic_memory(shadow_round);
			cout << "Scan p2m for dirty page - end " << get_time() << endl;
			printf("dynamic_pfn_map size: %d\n",dynamic_pfn_map.size());
		
		//	fprintf(stderr,"dirty roots: %d\n",dirty_roots.size());		

			print_static_alerts();

			page_map.clear();
			visited.clear();

#ifdef INV_LOG
			fprintf(inv_log,"[%s] End Shadow Round %d.\n",get_time(),shadow_round);	
			fflush(inv_log);		
#endif

			fprintf(stderr,"done shadow round %d\n",shadow_round);
			printf("done shadow round %d\n",shadow_round);
			shadow_round++;
			gettimeofday(&round_end,NULL);
#ifdef INV_LOG
			fprintf(inv_log,"Start: %d %d End: %d %d\n",round_start.tv_sec, round_start.tv_usec,round_end.tv_sec,round_end.tv_usec);
			float round_time = (round_end.tv_sec - round_start.tv_sec) * 1000000 + (round_end.tv_usec - round_start.tv_usec);
			fprintf(inv_log,"Time to Complete Round: %f\n",round_time);
			fflush(inv_log);			
#endif
			sleep(time_between);
		}
	}
	else
	{
#ifdef INV_LOG
		fprintf(inv_log,"Shadow Scan - Hypervisor Notifications\n");	
		fflush(inv_log);		
#endif
		/* recieve notifications from the hypervisor */

		/* TODO Add more pages to the watch list during rounds */

		uint32_t *pfn_watch_list;
	
		int i, pfn_watch_index;
		void *kick_handle;
		
		pfn_watch_list = (uint32_t *)malloc(p2m_size * sizeof(uint32_t));
		memset(pfn_watch_list, 0, p2m_size * sizeof(uint32_t));

		pfn_watch_index = 0;

		for(i = 0; i < p2m_size; i++)
		{
			multimap<unsigned long long, int>::iterator iter = static_symbol_map.find(i);
			if(iter != static_symbol_map.end())
			{
				/* add to pfn_watch_list */
				pfn_watch_list[pfn_watch_index] = i;
				pfn_watch_index++;
				
			}
			else
			{
				multimap<unsigned long long, entry_t>::iterator dyn_iter = dynamic_pfn_map.find(i);
				if(dyn_iter != dynamic_pfn_map.end())
				{
					/* add to pfn_watch_list */
					pfn_watch_list[pfn_watch_index] = i;
					pfn_watch_index++;
				}
			}
		}

		printf("pfn_watch_index size: %d\n",pfn_watch_index+1);

#ifdef INV_LOG
		fprintf(inv_log,"pfn_watch_index_size: %d\n",pfn_watch_index+1);	
		fflush(inv_log);		
#endif

		/* get handle for hypervisor notifications */
		kick_handle = xc_shadow_kick_prepare(xc_handle,dom);
		if(kick_handle == NULL)
		{
			fprintf(stderr, "Unable to get handle to shadow_kick\n");
			exit(-1);
		}

		int start = xc_shadow_kick_start(kick_handle, pfn_watch_list,pfn_watch_index+1,num_pages);
		if( start < 0)
		{
			fprintf(stderr, "Unable to start shadow_kick: %d\n",start);
			exit(-1);
		}

		bitmap = (unsigned long *)xg_memalign(PAGE_SIZE, ROUNDUP(BITMAP_SIZE, PAGE_SHIFT)); 

		if ( !bitmap )
		{
			fprintf(stderr,"Couldn't allocate bitmap array");
			exit(-1);
		}
	
		memset(bitmap, 0xff, BITMAP_SIZE);

		while(1)
		{
			xc_shadow_kick_wait(kick_handle);

		    printf("[%s] Start Shadow Round Hypervisor Notification %d.\n",get_time(),shadow_round);
			fprintf(stderr,"start shadow round notify %d\n",shadow_round);

#ifdef INV_LOG
			fprintf(inv_log,"[%s] Start Shadow Round Hypervisor Notification %d.\n",get_time(),shadow_round);
			fflush(inv_log);			
#endif

			static_count = 0;
			static_pages = 0;

#ifdef INV_LOG
			fprintf(inv_log,"Hypervisor Notification Occured - %d pages changed\n",num_pages);	
			fflush(inv_log);		
#endif
			printf("hypervisor notification occured, %d pages changed\n",num_pages);

			if ( xc_shadow_control(xc_handle, dom, 
								XEN_DOMCTL_SHADOW_OP_CLEAN, bitmap, 
								p2m_size, NULL, 0, &stats) != p2m_size )
			{
				fprintf(stderr,"Error flushing shadow PT");
				exit(-1);
			}

			fprintf(stderr,"stats: dirty: %d\n",stats.dirty_count);

#ifdef INV_LOG
			fprintf(inv_log,"\nNumber of Dirty Pages: %d\n\n",stats.dirty_count);
			fflush(inv_log);			
#endif

			for(i = 0; i < pfn_watch_index; i++)
			{
				unsigned long long dirty_pfn = pfn_watch_list[i];

				/* check for dirty page */
				if(test_bit(dirty_pfn,bitmap) == 1)
				{
					unsigned int addr = dirty_pfn << PAGE_SHIFT;

					STATIC_SYMBOL_CURRENT = -1;

					multimap<unsigned long long, int>::iterator iter = static_symbol_map.find(dirty_pfn);
					if(iter != static_symbol_map.end())
					{
						STATIC_SYMBOL_CURRENT = iter->second;
					}

					if(STATIC_SYMBOL_CURRENT != -1)
					{
						static_pages++;
						fetch_page(addr,0,shadow_round);
					//	printf("save_data_values STATIC_SYMBOL_CURRENT: %d\n",STATIC_SYMBOL_CURRENT);
						save_data_values(addr, shadow_round);
					}
					else
					{	
						multimap<unsigned long long, entry_t>::iterator iter = dynamic_pfn_map.find(dirty_pfn);
						if(iter != dynamic_pfn_map.end())
						{
							//printf("found changed dynamic symbol name: %s  type: %s\n",iter->second.pathname.c_str(), iter->second.sym_type);
					
							pair<multimap<unsigned long long,entry_t>::iterator,multimap<unsigned long long,entry_t>::iterator> entries;
							entries = dynamic_pfn_map.equal_range(dirty_pfn);
							int entrycount = 0;
							multimap<unsigned long long, entry_t>::iterator itr;

							for (itr=entries.first; itr!=entries.second; ++itr)
							{
								dirty_roots.push(itr->second);
								entrycount++;
							}
						}
					}
				}
			}

#ifdef INV_LOG
			fprintf(inv_log,"\nNumber of Dirty Roots: %d\n",dirty_roots.size());
			fprintf(inv_log,"Number of Static Pages: %d in %d pages\n\n",static_count,static_pages);
			fflush(inv_log);			
#endif	
			scan_dynamic_memory(shadow_round);	

			print_static_alerts();

			page_map.clear();
			visited.clear();

			printf("[%s] End Shadow Round Hypervisor Notifications %d.\n",get_time(),shadow_round);

			fprintf(stderr,"end shadow round notify %d\n",shadow_round);

#ifdef INV_LOG
			fprintf(inv_log,"[%s] End Shadow Round Hypervisor Notifications %d.\n",get_time(),shadow_round);	
			fflush(inv_log);		
#endif


			shadow_round++;
		}
	
		munmap(pfn_watch_list,p2m_size * sizeof(uint32_t));

	}

	gettimeofday(&end, NULL);
	printf("Completed measurement rounds = %d\n", (shadow_round - 1));
	printf("Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));

#ifdef INV_LOG
			fprintf(inv_log,"Completed measurement rounds = %d\n", (shadow_round - 1));
			fprintf(inv_log,"Total time taken = %d seconds\n",(end.tv_sec - start.tv_sec));	
			fflush(inv_log);		
#endif

	if(detection == 1)
	{
		fclose(fd_alerts);
	}

	fclose(inv_alerts);
	fclose(fd_stats);

	fclose(inv_log);
	munmap(memory_map,p2m_size * PAGE_SIZE);
	munmap(live_p2m,p2m_size * sizeof(xen_pfn_t));
	munmap(live_m2p,p2m_size * sizeof(xen_pfn_t));
}

void clear_static_data()
{
    vector<symbol_t>::iterator iter = static_symbol.begin();
    while(iter != static_symbol.end())
    {
        (*iter).values.clear();
        ++iter;
    }
}


/* Gets static memory pages */
void scan_static_memory(unsigned int mem_start, unsigned int mem_end, int msmt_round)
{
	unsigned int current_mem_start;
	int block_no =0;

	printf("Msmt round = %d\n", msmt_round);
	current_mem_start = mem_start;
	cout << "[" << get_time() << "] Begin - Static memory scan." << endl;

#ifdef INV_LOG
	fprintf(inv_log,"[%s] Begin - Static memory scan.\n",get_time());			
	fflush(inv_log);
#endif

	while(current_mem_start < mem_end)
	{
		clear_dmadata_buffer();
		//printf("Static memory scan ... Requesting page 0x%x\n", current_mem_start);

		fetch_page(current_mem_start, block_no, msmt_round);
		// JEFF BUFFER printf("monitor dmadata_buffer: %s\n",dmadata_buffer);
		save_data_values(current_mem_start, msmt_round);

		//Increment the block_no and also msmt_round if required
		if((current_mem_start + SLAB_SIZE) == mem_end)
		{
			// Start static memory traversal again
			STATIC_SYMBOL_CURRENT = 0;
		}

		current_mem_start = current_mem_start + SLAB_SIZE;
		//printf("current_mem_start: 0x%x mem_end: 0x%x",current_mem_start,mem_end);
        block_no++;

		if(exit_now == 1)
		{
			printf("Stopping static memory scan ... \n");
			return;
		}
	}
	
	cout << "[" << get_time() << "] End - Static memory scan." << endl;

#ifdef INV_LOG
	fprintf(inv_log,"[%s] End - Static memory scan.\n",get_time());			
	fflush(inv_log);
#endif

	// JEFF changed msmt_round from > 2 to > 1
	if((detection == 1) && (msmt_round > 1))
	{	
        print_static_alerts();
	}
}

void print_static_alerts()
{
	cout << "printing static alerts" << endl;
	for(int i=0; i<static_symbol.size(); i++)
	{
		if(static_symbol[i].alert)
		{
			symbol_t s = static_symbol[i];

			fprintf(fd_alerts,"[Static Alert] addr=0x%llx,val=%llx,name=%s\n",s.sym_addr,get_vector_value(s.values),s.sym_name);
			//fflush(fd_alerts); // Was crashing the program
			static_symbol[i].alert = false;
			static_symbol[i].reported = true;
		}
	}
	fflush(fd_alerts);
	cout << "Finished printing static alerts" << endl;
}

unsigned long long get_vector_value(vector<unsigned char> bytes)
{
	char strbytes[10];
	unsigned long long ret_val;
	
	//assert(bytes.size() >= 1);

	if(bytes.size() <= 8)
	{
		for(int i=0; i<bytes.size(); i++)
		{
			strbytes[i] = bytes[i];
		}	
	}

	if(bytes.size() == 1)
		ret_val = *(unsigned char *)(strbytes);	
	else if(bytes.size() == 2)
		ret_val = *(unsigned short *)(strbytes);	
	else if(bytes.size() == 4)
		ret_val = *(unsigned int *)(strbytes);	
	else if(bytes.size() == 8)
		ret_val = *(unsigned long long *)(strbytes);	
	else
		ret_val = 9999; //0x270F

	return(ret_val);
}

void print_stats(int msmt_round, int num_secs)
{
		
    //fprintf(fd_stats,"Mode=%d, Msmt round =%d\n",detection, msmt_round);
    fprintf(fd_stats,"%d ",total_objects);
    //fprintf(fd_stats,"Objects seen before=%d\n",same_object_cnt);
    fprintf(fd_stats,"%d ",total_symbols);
    fprintf(fd_stats,"%d ", obj_func_ptr);
    fprintf(fd_stats,"%d ", total_func_ptr);
    fprintf(fd_stats,"%d ",num_secs);
    fprintf(fd_stats,"%d\n", inconsistent_cnt);
	fflush(fd_stats);
                                                                                                                                               
}


void clear_dmadata_buffer()
{
	for(int i=0; i< SLAB_SIZE; i++)
	{
		dmadata_buffer[i] = 0;
	}
}

/* To be part of the init_roots set, the symbol has to be a pointer to a composite type 
   i.e. a struct/union pointer or pointer array */
vector<int> startset;

bool in_startset(int srchindex)
{
	for(int i=0; i < startset.size(); i++)
	{
		if(startset[i] == srchindex)
		{	
			return(true);
		}
	}	
	return(false);
}

void populate_start_set()
{
	short num_roots=0;
	cout << "Populate_start_set : num nodes before= "<< listq.size() << endl;
	for(int i=0; i< static_symbol.size(); i++)
	{
		//Check that it is a pointer and not a function pointer
		//if((strstr(static_symbol[i].sym_type,"(*)") == NULL) && (strstr(static_symbol[i].sym_type,"*") != NULL)) 
		/*if(!func_ptr(static_symbol[i].sym_type))
		{
			if(((strstr(static_symbol[i].sym_type,"struct") != NULL) || (strstr(static_symbol[i].sym_type,"union"))) && 
				(strstr(static_symbol[i].sym_type,"*") != NULL))
			{*/
			//printf("static_symbol: %s\n",static_symbol[i].sym_type);

			if(ptr_to_composite_type(static_symbol[i].sym_type))
			{
				//printf("true\n");
				//This is either a struct/union pointer or pointer array
				//Add to set of initial roots
				init_roots.push_back(i);

				entry_t ent;
				ent.sym_addr = static_symbol[i].sym_addr;	
				strcpy(ent.sym_type, static_symbol[i].sym_type);
				ent.sym_id = static_symbol[i].sym_addr;
				if(strstr(ent.sym_type,ATTRIBUTE_STRING) != NULL)
				{
					strcpy(ent.sym_attr, ent.sym_type);
				}
				ent.pathname = static_symbol[i].sym_name;
				ent.pathtype= static_symbol[i].sym_type;

				// JEFF PATHADDR
				std::stringstream pathout;
				pathout << "0x" << hex << static_symbol[i].sym_addr;
				ent.pathaddr.push_back(pathout.str());

			//	printf("sym_addr: 0x%x ",static_symbol[i].sym_addr);
			//	printf("pathaddr: %s\n",ent.pathaddr.c_str());

				ent.root = true;
				if(ll_head(ent.sym_type))
				{
					//Add node to linked list queue
					listq.push(ent);
					startset.push_back(i);
				} 
#ifdef LOG
				//JEFF causing segment fault comment out
				if(start_logging)
				{
					char logbuffer[MAX_LINE_INPUT_SIZE];

					//sprintf(logbuffer,"Adding root - %s \taddr=0x%llx, type=%s", static_symbol[i].sym_name, 
					//				static_symbol[i].sym_addr, static_symbol[i].sym_type);
					//logln(logbuffer);
				
					/*if(++num_roots	> 1)
					{
						return;
					}*/
				}
#endif
			}
			//}
		//}
	}
	cout << "Populate_start_set : num nodes = "<< listq.size() << endl;
}

/*bool func_ptr(char * s)
{
	//char func_def[MAX_FIELD_NAME];
	char func_def[MAX_LINE_INPUT_SIZE];

	if(s != NULL)
	{
		strcpy(func_def, s);
		trim(func_def);
		
		int len = strlen(func_def);

		if((strstr(func_def,"(*)") != NULL) ||
		  (((strstr(func_def,"(") != NULL)  && (strstr(func_def,")") != NULL))   && (strstr(func_def,ATTRIBUTE_STRING) == NULL)))
		{
			return(true);
		}
	}
	return(false);
} */

void get_ptrvalues_from_static(int msmt_round)
{
	//If the static symbol is a regular pointer, get the actual data value
	//from the pointed location
	for(int i=0; i<static_symbol.size(); i++)
	{
		handle_reg_ptr_data(static_symbol[i], msmt_round);
	}		
}

void handle_reg_ptr_data(symbol_t current_sym, int msmt_round)
{
	//if((strcmp(current_sym.sym_type,"(*)(") != 0) && (strcmp(current_sym.sym_type,"*") == 0))
	if((!func_ptr(current_sym.sym_type)) && (strcmp(current_sym.sym_type,"*") == 0))
	{
		//Also skip struct/union pointers
		if((strcmp(current_sym.sym_type,"struct") != 0) && (strcmp(current_sym.sym_type,"union") != 0))
		{
			char val_addr[4];

			//This is a regular pointer
			unsigned int obj_addr;
			int last_entry = current_sym.values.size();

			for(int k=0; k<4; k++)
			{
				val_addr[k] = current_sym.values[k];
			}

			obj_addr = *(unsigned int *)(val_addr);

			//assert(obj_addr > 0);
			//assert(obj_addr <= 0xf8000000);
				

			//Convert the virtual address to physical address
			//obj_addr = virt_to_phys(obj_addr);

			//Check if this page already exists in page map	
			unsigned char current_page[SLAB_SIZE];

			unsigned long long page_addr = page_addr_from_sym_addr(obj_addr);

			if(valid_page(page_addr))
			{
				page_t new_page = get_page(page_addr, msmt_round);
				//strcpy((char *)current_page, (const char *)new_page.page_data);
				copy_bytes(new_page.page_data, current_page,SLAB_SIZE,0);

				int offset = page_addr - obj_addr;
				int obj_size;	
				string obj_type;

				//Look at pointer type and decide what it should point to
				if(strstr(current_sym.sym_type,"char") != NULL)
				{
					//This is a character pointer
					obj_type = "char";
					obj_size = sizeof(char);
				}
				else if(strstr(current_sym.sym_type,"int") != NULL)
				{
					//This is a int pointer
					obj_type = "int";
					obj_size = sizeof(int);
				}
				else if(strstr(current_sym.sym_type,"long long") != NULL)
				{
					//This is a long long pointer
					obj_type = "long long";
					obj_size = sizeof(long long);
				}
				else if(strstr(current_sym.sym_type,"long") != NULL)
				{
					//This is a long long pointer
					obj_type = "long";
					obj_size = sizeof(long);
				}
				else if(strstr(current_sym.sym_type,"short") != NULL)
				{
					//This is a long long pointer
					obj_type = "short";
					obj_size = sizeof(short);
				}
				else if(strstr(current_sym.sym_type,"double") != NULL)
				{
					//This is a long long pointer
					obj_type = "double";
					obj_size = sizeof(double);
				}

				vector<unsigned char> bytes;
				for(int k=0; k<current_sym.sym_size; k++)
				{
					bytes.push_back(*(unsigned char *)(current_page + offset + k));
				}

				multimap<unsigned long long, symbol_t>::iterator iter = dynamic_symbol.find(obj_addr);

				if(iter != dynamic_symbol.end())
				{
					//Object not found
					//Create new  symbol and add into this map
					if(detection == 0)
					{
						symbol_t s;

						strcpy(s.sym_name, "*");
						strcat(s.sym_name, current_sym.sym_name);
						//Arati - enable this later
						//s.values.push_back(bytes);
						s.sym_id = obj_addr;
						s.sym_addr = obj_addr;	

						dynamic_symbol.insert(make_pair(obj_addr,s));
					}
				}
				else
				{
					//Object found - Check if object type matches
					if (strcmp((iter->second).sym_type, obj_type.c_str()) == 0)
					{
						//Treat as same object
						//If in learning, append value of current pass
						if(detection == 0)
						{	
							//Arati - enable this later
							//(iter->second).values.push_back(bytes);
						}
					}
					else
					{
						//if in learning phase, delete previos object, add new object
						if(detection == 0)
						{
							//Need to assert that there is only one object here
							assert(iter == dynamic_symbol.upper_bound(obj_addr));
							dynamic_symbol.erase(iter);

							//Create new  symbol and add into this map
							symbol_t s;

							strcpy(s.sym_name, "*");
							strcat(s.sym_name, current_sym.sym_name);
							//Arati - enable this later
							//s.values.push_back(bytes);
							s.sym_id = obj_addr;
							s.sym_addr = obj_addr;	

							dynamic_symbol.insert(make_pair(obj_addr,s));
						}

					}
				}
			}
		}
	}
}

void open_filter_file(int msmt_round, string prefix)
{
	if(detection == 0)
	{
		std::stringstream strfilter;
		strfilter << msmt_round;

		string filename = "snapshots/" + prefix;
		filename.append(strfilter.str());
		filename.append(".ir.dtrace");

		fd_filter = fopen(filename.c_str(),"a");

		if(fd_filter == NULL)
		{
			cout << "Cannot open file " << filename.c_str() << endl;
			exit(-1);
		}
	}
}

void close_filter_file()
{
	if(detection == 0)
	{
		fclose(fd_filter);
	}
}

bool declared(string prefix, int msmt_round)
{
	std::stringstream strfilter;
	struct stat s;

	strfilter << msmt_round;

	string filename = "snapshots/";
	filename.append(prefix);
	filename.append(".ir.decls");

	//cout << "stating file " << filename << endl;
	int ret = stat(filename.c_str(),&s);

	if(stat(filename.c_str(),&s)  == -1)
	{
		return(false);
	}
	else
	{
		return(true);
	}
}

void scan_dynamic_memory(int msmt_round)
{
#ifdef LOG
	log_init();
#endif
	bool print_daikon_fields = false;

	/* Locate pointers in the static_symbols vector, add to base set of roots */
	queue<entry_t> bfs_roots; 
	string filter; 
	string prefix;	

	if(shadow == 0 || msmt_round == 1)
	{
		/* Populates the begin bfs nodes from init_roots set */
		bfsnodes_from_initroots(bfs_roots);
	}
	else
	{
		/* Populates the begin bfs nodes from dirty_roots set */
		while(!dirty_roots.empty())
		{
			bfs_roots.push(dirty_roots.front());
			dirty_roots.pop();
		}
	}
	/* Populate the values from the static set in the hash map for regular pointers */
	//get_ptrvalues_from_static(msmt_round);
	//printf("[%s] Finished populating pointer values from static data...\n", get_time());

	/* Set flag to print daikon fields in learning mode */
	if(detection ==0)
		print_daikon_fields = true;

	cout << "[" << get_time() << "] Begin - Dynamic memory scan " << endl;

#ifdef INV_LOG
	fprintf(inv_log,"[%s] Begin - Dynamic memory scan.\n",get_time());			
	fflush(inv_log);
#endif

	printf("bfs roots size: %d visited size: %d def_map: %d\n",bfs_roots.size(),visited.size(),def_map.size());

	while((!bfs_roots.empty()) &&  (total_objects < MAX_OBJECTS_TRAVERSED))
	{
		//Remove an element from the front of the queue
		entry_t node;
		unsigned long long sym_addr;
		unsigned long long page_addr;
		unsigned long long sym_id;
		string sym_type;
		string sym_attr;
		string sym_pathname;
		string sym_pathtype;
		vector<string> sym_pathaddr;  //JEFF PATHADDR
		bool sym_root;
		bool same_object = false;

		node = bfs_roots.front();
		sym_addr = node.sym_addr;
		sym_type = node.sym_type;
		sym_id = node.sym_id;
		sym_attr= node.sym_attr; //Take care of container offsets
		sym_pathname = node.pathname;
		sym_pathtype = node.pathtype;
		sym_pathaddr = node.pathaddr; //JEFF PATHADDR
		sym_root = node.root;
		total_objects++;

		int found = 0;
		bool consider_llhead = false;

		if(sym_root)
			consider_llhead = true;

		if(shadow == 1 && msmt_round == 1)
		{
			
			unsigned long long pfn = virt_to_pfn(sym_addr);
			dynamic_pfn_map.insert(make_pair(pfn,node));
			dynamic_path_map.insert(make_pair(sym_pathname,node));
		}
		else if(shadow == 1)
		{
			unsigned long long sym_pfn = virt_to_pfn(sym_addr);

			multimap<unsigned long long, entry_t>::iterator iter = dynamic_pfn_map.find(sym_pfn);
			if(iter != dynamic_pfn_map.end())
			{
				pair<multimap<unsigned long long,entry_t>::iterator,multimap<unsigned long long,entry_t>::iterator> entries;
				entries = dynamic_pfn_map.equal_range(sym_pfn);
				multimap<unsigned long long, entry_t>::iterator itr;

				entry_t checknode;
	
				for (itr=entries.first; itr!=entries.second; ++itr)
				{
					if(itr->second.sym_addr == sym_addr){
						checknode = itr->second;
						found = 1;
						break;
					}	
				}
				//printf("Popping Node at addr= 0x%llx of type=%s attr=%s ID=0x%llx", sym_addr, sym_type.c_str(), sym_attr.c_str(), sym_id);
				if(found == 1)
				{
					/* symbol found already in mapping, check value? */
					//printf("already found: %s\n",sym_pathname.c_str());
				}
				else
				{
					//printf("symbol not in mapping, add it\n");
					dynamic_pfn_map.insert(make_pair(sym_pfn,node));

					map<string, entry_t>::iterator piter = dynamic_path_map.find(sym_pathname);

					if(piter != dynamic_path_map.end())
					{
						//printf("symbol for path: %s already exists\n",sym_pathname.c_str());
					//	printf("pathaddr for current:   %s\n",sym_pathaddr.c_str());
					//	printf("pathaddr for previous: %s\n",(piter->second).pathaddr.c_str());

						vector<string> current = sym_pathaddr;
						vector<string> previous = piter->second.pathaddr;
					
						if(current.size() == previous.size())
						{
							int i;
							bool change = true;
							for(i = 0; i < current.size()-1; i++)
							{
								if(strcmp(current[i].c_str(),previous[i].c_str()) != 0)
								{
									change = false;	
									break;
								}
							}
							
							if(change == true && strcmp(current[i].c_str(),previous[i].c_str()) == 0)
							{	
								/* remove previous node from dynamic pfn map */
								entry_t prev_node = piter->second;

								unsigned long long pfn = virt_to_pfn(prev_node.sym_addr >> PAGE_SHIFT);
							
								multimap<unsigned long long, entry_t>::iterator citer = dynamic_pfn_map.find(pfn);
								while(citer != dynamic_pfn_map.end())
								{
									/* compare path of addresses since the assumption is this is unique */
									if(citer->second.pathaddr == prev_node.pathaddr)
									{
										dynamic_pfn_map.erase(citer);
										break;
									}
									citer++;
								}
							}
						}
					}
					else
						dynamic_path_map.insert(make_pair(sym_pathname,node));
				}
			}
			else
			{
				//printf("iter equals dynamic_pfn_map end add to mapping\n");
				dynamic_pfn_map.insert(make_pair(sym_pfn,node));
			}
		}

#ifdef LOG
		if(start_logging)
		{	
			char logbuffer[MAX_LINE_INPUT_SIZE];
			sprintf(logbuffer,"Popping Node at addr= 0x%llx of type=%s attr=%s ID=0x%llx", sym_addr, sym_type.c_str(), sym_attr.c_str(), sym_id);
			fprintf(inv_log,"path: %s\n",sym_pathname.c_str());
			
			logln(logbuffer);
		}
#endif 

#ifdef INV_LOG
		fprintf(inv_log,"[%s] Popping Node at addr= 0x%llx of type=%s attr=%s ID=0x%llx\n", get_time(), sym_addr, sym_type.c_str(), sym_attr.c_str(), sym_id);
	//	fprintf(inv_log,"path: %s\n",sym_pathname.c_str());
		fflush(inv_log);			
#endif
		
		//
		if((strstr(node.sym_attr,ATTRIBUTE_STRING) != NULL) &&  
		  (strstr(node.sym_attr,CONTAINER_ATTRIBUTE)!= NULL))
		{
			sym_type = get_container_type(sym_attr);	
		}

		filter = trim((char *)sym_type.c_str());

		map<string, int>::iterator ts_iter = type_stats.find(filter);
		if( ts_iter == type_stats.end())
		{
			type_stats.insert(make_pair(filter,1));
		}
		else
		{
			ts_iter->second++;
		}

		//Intermittently show some progress
		if((bfs_roots.size() % 50)  == 0)
		{
			cout << "#";
		}
		
		char strprefix[MAX_LINE_INPUT_SIZE];
		//char *strprefix = (char *)malloc(MAX_LINE_INPUT_SIZE);
		strcpy(strprefix,filter.c_str());
	
		// JEFF added fix for bio_destructor_t
		int spacei;
		int spacecount = 0;
		for(spacei=0;spacei<strlen(strprefix);spacei++){
			if(strprefix[spacei] == ' ')
				spacecount++;
		}
		
		//printf("spacecount: %d\n",spacecount);
	
		prefix = strtok(strprefix, " "); 
		if(spacecount > 0)
			prefix = strtok(NULL, " ");

		open_filter_file(msmt_round, prefix);

		//Remove const keyword if it exists at the end
		remove_const_keyword(sym_type);
		trim((char *)sym_type.c_str());

		/* Find the definition of the node type in definition map */
		map<string,definition_t>::iterator iter = def_map.find(sym_type.c_str());

	//	printf("sym_type: %s bfs_roots size: %d\n",sym_type.c_str(),bfs_roots.size());			

//		printf("Popping Node at addr= 0x%llx of type=%s attr=%s ID=0x%llx", sym_addr, sym_type.c_str(), sym_attr.c_str(), sym_id);
		if(iter != def_map.end())
		{
			//Found definition 
			vector<field_t> fields = iter->second;

			for(int i=0; i<fields.size(); i++)
			{
				field_t fld = fields[i];
				string fld_type = fld.type;
				string fld_name = fld.name;
				int fld_offset = fld.offset;
				string container_fld_name = get_container_field_name(sym_attr);	
				
				if(strncmp(fld_name.c_str(),container_fld_name.c_str(),strlen(container_fld_name.c_str())) ==0)
				{
					sym_addr = sym_addr - fld_offset;
					sym_id = sym_addr;
#ifdef LOG
					if(start_logging)
					{
						char logbuffer[MAX_LINE_INPUT_SIZE];
						sprintf(logbuffer,"Container type=%s, Adjusting addr=0x%llx,Changing id=0x%llx\n",sym_type.c_str(),sym_addr,sym_id);
						logln(logbuffer);
					}
#endif
					break;
				}
			}

#ifdef PRINT_DAIKON_TRACES

			if(!declared(prefix, msmt_round))
			{
				cout << "Creating declaration for " << prefix << endl;
				//Create Daikon declaration 
				create_daikon_declaration(fields, prefix, msmt_round, sym_type);
			}
#endif
			//Find the page this data structure belongs to
			page_addr = page_addr_from_sym_addr(sym_addr);

			//Check if this page already exists in page map	
			unsigned char current_page[SLAB_SIZE];

			if(valid_page(page_addr)) // Indent this later
			{
				page_t new_page = get_page(page_addr, msmt_round);
				copy_bytes(new_page.page_data, current_page, SLAB_SIZE, 0);

				int offset = sym_addr - page_addr;

				bool considered = false;
				int page_tracker = offset;
				short page_count = 1;

#ifdef PRINT_DAIKON_TRACES
				if(print_daikon_fields)
				{
					fprintf(fd_filter, "%s:", prefix.c_str());
					fprintf(fd_filter, "0x%llx:", sym_addr);
					fprintf(fd_filter, "%s\n", sym_pathname.c_str());
				}
#endif
			
#ifdef LOG
				if(start_logging)
				{
					char logbuffer[MAX_LINE_INPUT_SIZE];
					sprintf(logbuffer,"%0x%x 0x%x",sym_addr, page_addr);
					log("Node addr, Page addr: "); 
					logln(logbuffer);
					log("Node name: "); 
					logln((char *)sym_pathname.c_str());
					log("Node type: "); 
					logln((char *)sym_pathtype.c_str());
				}
#endif
				for(int i=0; i<fields.size(); i++)
				{
					fld_ent_t current_fld;

					field_t fld = fields[i];
					string fld_type = fld.type;
					string fld_name = fld.name;
					int fld_offset = fld.offset;
					int fld_size = fld.size;

					//if(ptr_to_composite_type(fld_type))
					//	printf("Field type: %s TRUE\n",fld_type.c_str());
					//else
					//	printf("Field name: %s FALSE\n",fld_name.c_str());
//JEFF logging
/*#ifdef LOG
					if(start_logging)
					{
						char logbuffer[MAX_LINE_INPUT_SIZE];
						log("Field type: "); 
						logln((char *)fld_type.c_str());
						log("Field name: "); 
						logln((char *)fld_name.c_str());
						sprintf(logbuffer,"%d %d",fld_offset, fld_size);
						log("Field offset, Field size: "); 
						logln(logbuffer);
					}
#endif*/

					total_symbols++;

					if(func_ptr((char *)fld_type.c_str()))
					{
					//	printf("function ptr: %s %s\n",fld_name.c_str(),fld_type.c_str());
						total_func_ptr++;
						if(!considered)
						{
							obj_func_ptr++;
							considered = true;
						}
					}

					//Look at each field in the object name,type,offset,size 
					//If type is an array, then size contains the total size of all array elements
					if((strstr(fld_type.c_str(),"[") != NULL) && (!func_ptr((char *)fld_type.c_str())))
					{
						//Type is an array
						int index = get_array_index(fld_type.c_str());

						if((fld_size == 0) && (index ==0))
						{ // type[0] case
							fld_size = 1;
							index = 1;
						}
/* JEFF logging
#ifdef LOG
						if(start_logging)
						{
							char logbuffer[MAX_LINE_INPUT_SIZE];
							sprintf(logbuffer,"%d %d",fld_size, index);
							log("Array Type: Field size, Index: "); 
							logln(logbuffer);
						}
#endif */

						int arr_element_size = (fld_size / index);
						if(print_daikon_fields)
						{
#ifdef PRINT_DAIKON_TRACES
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								fprintf(fd_filter,"%s[]\n", fld_name.c_str());
								fprintf(fd_filter,"[");

								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
									fprintf(fd_filter,"\"");
							}
#endif
						}
						else
						{
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								current_fld.field_name = fld_name + "[]";
								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
								{
									current_fld.str_field_value = "[";
									current_fld.isstrfield = true;
									current_fld.str_field_value.append("\"");
								}
								else
								{
									current_fld.numeric_field_value = "[";
									current_fld.isstrfield = false;
								}
							}

						}

						//Create index number of symbols
						bool null_found = false;
						for(int j=0; j<index; j++) 
						{
							int fld_begin;
							if((offset + fld_offset + (j*arr_element_size)) > (page_count * SLAB_SIZE)) //Fetch a new page
							{
								if(valid_page(page_addr + (page_count * SLAB_SIZE)))
								{
									new_page = get_page(page_addr + (page_count * SLAB_SIZE), msmt_round);
									copy_bytes(new_page.page_data, current_page, SLAB_SIZE, 0);
									fld_begin = (offset + fld_offset + (j*arr_element_size)) % (SLAB_SIZE);
									++page_count;
								}
							}
							else if ((offset + fld_offset + (j*arr_element_size)) < SLAB_SIZE) //Within the first page
							{
								fld_begin = offset + fld_offset + (j*arr_element_size);
							}
							else //Within the fetched page
							{
								fld_begin = (offset + fld_offset + (j*arr_element_size)) % (SLAB_SIZE);
							}

							symbol_t current_sym;
							vector<unsigned char> current_value;


							for(int k=0;k < arr_element_size; k++)
							{
								current_value.push_back(*(unsigned char *)(current_page + fld_begin + k));
							}
							if(print_daikon_fields)
							{
#ifdef PRINT_DAIKON_TRACES
								if(strstr(fld_type.c_str(),"union") == NULL)
								{
									if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
									{
										if(*(unsigned char *)(current_page + fld_begin) == '\0')
										{
											null_found = true;
										}
										if(!null_found)
										{
											fprintf(fd_filter,"%c", *(unsigned char *)(current_page + fld_begin));
										}
									}
									else
									{
										fprintf(fd_filter,"%lld ", get_vector_value(current_value));
									}
								}
#endif
							}	
							else
							{
								if(strstr(fld_type.c_str(),"union") == NULL)
								{
									if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
									{
										if(*(unsigned char *)(current_page + fld_begin) == '\0')
										{
											null_found = true;
										}
										if(!null_found)
										{
											//*(unsigned char *)(current_page + fld_begin);
											stringstream tmp;
											tmp <<  *(unsigned char *)(current_page + fld_begin);
											current_fld.str_field_value.append(tmp.str());
										}
									}
									else
									{
										std::stringstream tmp;
										tmp <<  get_vector_value(current_value);
										current_fld.numeric_field_value.append(tmp.str());
										current_fld.numeric_field_value.append(" ");
									}
								}
							}
					
							//If field is a pointer to struct/union, create a entry_t object and add to bfs_roots queue
							if(ptr_to_composite_type(fld_type))
							{
								//If pointer, then check visited map
								//If not in visited map, make an entry_t object and push in BFS_ROOTS queue
								unsigned long long tmp_addr = *(unsigned int *)(current_page + fld_begin);

								if(tmp_addr != 0)
								{	
									map<unsigned long long, char>::iterator iter = visited.find(tmp_addr);

									if((consider_llhead) && (ll_head((char *)fld_type.c_str()) || ll_head((char *)fld_name.c_str())))
									{
										entry_t e;

										e.sym_addr = tmp_addr; 
										//e.sym_addr = sym_addr; 
										e.sym_id = sym_id;
										e.field_offset = fld_offset;

										string container_type = get_container_type(fld_type);
										strcpy(e.sym_type, container_type.c_str());			
										e.pathname = sym_pathname + PATH_SEPARATOR + fld_name; 
										e.pathtype = sym_pathtype + PATH_SEPARATOR + fld_type; 
	
										// JEFF PATHADDR
										std::stringstream pathout;
										pathout << "0x" << hex << e.sym_addr;
										e.pathaddr.push_back(pathout.str());

										e.sym_attr[0]='\0';
										if(strstr(fld_type.c_str(),ATTRIBUTE_STRING) != NULL)
										{
											strcpy(e.sym_attr, fld_type.c_str());
										}
										e.root = false;
										listq.push(e);
									}

								 	//Ignore if pointer points to itself or to the field just before it (for list_head's)
                                	if((tmp_addr != (page_addr+ offset + fld_offset + (j*arr_element_size))) && (tmp_addr != (page_addr+ offset + fld_offset + (j*arr_element_size)- 4)))
                                	{
										if(iter == visited.end())
										{	
											entry_t e;

											e.sym_addr = tmp_addr; 
											e.sym_id = sym_id;

											string container_type = get_container_type(fld_type);
											strcpy(e.sym_type, container_type.c_str());			
											e.pathname = sym_pathname + PATH_SEPARATOR + fld_name; 
											e.pathtype = sym_pathtype + PATH_SEPARATOR + fld_type; 

											// JEFF PATHADDR
											std::stringstream pathout;
											pathout << "0x" << hex << e.sym_addr;
											e.pathaddr.push_back(pathout.str());

											if(strstr(fld_type.c_str(),ATTRIBUTE_STRING) != NULL)
											{
												strcpy(e.sym_attr, fld_type.c_str());
												//Begin of the linked list should be put in visited to avoid cases
												//linked list of type a points to linked list of type b
												visited.insert(make_pair((sym_addr + fld_offset), 1));
											}
											else
											{
												e.sym_attr[0]='\0';
											}
//#ifdef WHITE_LIST
//											if(in_white_list(e.sym_type))
//#else
											if(!in_black_list(e.sym_type))
//#endif											
											{
												if(!valid_addr(e.sym_addr))
												{
													if(e.sym_addr < 0xf8000000)
													{
														fprintf(fd_dangle,"[%s]\n", e.pathname.c_str());  
														inconsistent_cnt++;
													}
												}

												if(valid_addr(tmp_addr))
												{
													if(!in_static_area(tmp_addr)) 
													{	
														if(strcmp(e.sym_type,"struct list_head") != 0)
														{
															e.root = false;
															visited.insert(make_pair(tmp_addr, 1));	

															if(shadow == 0 || msmt_round == 1)
																bfs_roots.push(e);
															else {
																if(found == 0) {
																	if(test_bit(e.sym_addr >> PAGE_SHIFT,bitmap))
																		bfs_roots.push(e);
																	else
																		printf("page not dirty\n");
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
						if(print_daikon_fields)
						{	
#ifdef PRINT_DAIKON_TRACES
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
									fprintf(fd_filter,"\"");
								fprintf(fd_filter,"]\n");
								fprintf(fd_filter,"1\n");
							}
#endif
						}
						else
						{
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL)) //Char array
									current_fld.str_field_value.append("\"");

								if(current_fld.isstrfield)
									current_fld.str_field_value.append("]");
								else
									current_fld.numeric_field_value.append("]");
							}
						}
					}
					else
					{
						int fld_begin;
						if((offset + fld_offset) > (page_count *SLAB_SIZE)) //Fetch a new page
						{
							if(valid_page(page_addr + (page_count * SLAB_SIZE)))
							{
								new_page = get_page(page_addr + (page_count * SLAB_SIZE), msmt_round);
								copy_bytes(new_page.page_data, current_page, SLAB_SIZE, 0);
								fld_begin = (offset + fld_offset) % (SLAB_SIZE);	
								++page_count;
							}
						}
						else if ((offset + fld_offset) < SLAB_SIZE) //Within the first page
						{
							fld_begin = offset + fld_offset;
						}
						else //Within the fetched page other than the first page
						{
							fld_begin = (offset + fld_offset) % (SLAB_SIZE);
						}

						//If primitive field, then store value
						symbol_t current_sym;
						vector<unsigned char> current_value;

						for(int k=0;k < fld_size; k++)
						{
							current_value.push_back(*(unsigned char *)(current_page + fld_begin + k));
						}

						//Corresponds to Daikon trace
						if(print_daikon_fields)
						{
#ifdef PRINT_DAIKON_TRACES
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								fprintf(fd_filter,"%s\n", fld_name.c_str());
								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL))
									fprintf(fd_filter,"\"");
									fprintf(fd_filter,"%lld", get_vector_value(current_value));

								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL))
									fprintf(fd_filter,"\"");

									fprintf(fd_filter,"\n1\n");
							}
#endif
						}	
						else
						{
							if(strstr(fld_type.c_str(),"union") == NULL)
							{
								current_fld.field_name = fld_name;
								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL))
								{
									current_fld.isstrfield = true;
									current_fld.str_field_value.append("\"");
									stringstream tmp;
									tmp << get_vector_value(current_value);
									current_fld.str_field_value.append(tmp.str());
								}
								else
								{
									current_fld.isstrfield = false;
									stringstream tmp;
									tmp << get_vector_value(current_value);
									current_fld.numeric_field_value.append(tmp.str());
								}

								if((strstr(fld_type.c_str(),"char") != NULL) && (strstr(fld_type.c_str(),"*") == NULL))
									current_fld.str_field_value.append("\"");
							}
						}

						//If field is a pointer to struct/union, create a entry_t object and add to bfs_roots queue
						if(ptr_to_composite_type(fld_type))
						{
							unsigned long long tmp_addr = *(unsigned int *)(current_page + fld_begin);
						//	printf("tmp_addr: 0x%x\n",tmp_addr);

							if(tmp_addr != 0)
							{
								string container_type = get_container_type(fld_type);

								//If pointer, then check visited map
								//If not in visited map, make an entry_t object and push in BFS_ROOTS queue
								map<unsigned long long, char>::iterator iter = visited.find(tmp_addr);

								if((consider_llhead) && (ll_head((char *)fld_type.c_str()) || ll_head((char *)fld_name.c_str())))
								{
									entry_t e;

									e.sym_addr = tmp_addr;
									e.field_offset = fld_offset;
									e.sym_id = sym_id;

									strcpy(e.sym_type, container_type.c_str());			
									e.pathname = sym_pathname + PATH_SEPARATOR + fld_name; 
									e.pathtype = sym_pathtype + PATH_SEPARATOR + fld_type; 

									// JEFF PATHADDR
									std::stringstream pathout;
									pathout << "0x" << hex << e.sym_addr;
									e.pathaddr.push_back(pathout.str());

									e.sym_attr[0]='\0';
									if(strstr(fld_type.c_str(),ATTRIBUTE_STRING) != NULL)
									{
											strcpy(e.sym_attr, fld_type.c_str());
									}
									e.root = false;
									listq.push(e);
								}
							
								//Ignore if pointer points to itself or to the field just before it (for list_head's)
								if((tmp_addr != (sym_addr + fld_offset)) && (tmp_addr != (sym_addr + fld_offset - 4)))
								{	
									if(iter == visited.end())
									{	
										entry_t e;

										e.sym_addr = tmp_addr;
										e.sym_id = sym_id;

										strcpy(e.sym_type, container_type.c_str());			
										e.pathname = sym_pathname + PATH_SEPARATOR + fld_name; 
										e.pathtype = sym_pathtype + PATH_SEPARATOR + fld_type; 

										// JEFF PATHADDR
										std::stringstream pathout;
										pathout << "0x" << hex << e.sym_addr;
										e.pathaddr.push_back(pathout.str());

										if(strstr(fld_type.c_str(),ATTRIBUTE_STRING) != NULL)
										{
											strcpy(e.sym_attr, fld_type.c_str());

											//Begin of the linked list should be put in visited to avoid cases
											//linked list of type a points to linked list of type b
											visited.insert(make_pair((sym_addr + fld_offset), 1));
										}
										else
										{
											e.sym_attr[0]='\0';
										}

//#ifdef WHITE_LIST
		//								if(in_white_list(e.sym_type))
//#else
										if(!in_black_list(e.sym_type))
//#endif											
										{
											if(!valid_addr(e.sym_addr))
											{
												//if((detection == 1) && (msmt_round == 1))
												if(e.sym_addr < 0xf8000000)
												{
													fprintf(fd_dangle,"[%s]\n", e.pathname.c_str());  
													inconsistent_cnt++;
												}
											}

											if(valid_addr(tmp_addr))
											{	
												if(!in_static_area(tmp_addr)) //C45
												{
													assert((e.sym_addr == 0x000000) ||valid_addr(e.sym_addr));
													if(strcmp(e.sym_type,"struct list_head") != 0)
													{
														visited.insert(make_pair(tmp_addr, 1));		
														e.root = false;	
														//printf("push node 2 %s %s\n",fld_name.c_str(),fld_type.c_str());
														if(shadow == 0 || msmt_round == 1)
															bfs_roots.push(e);
														else {
															if(found == 0) {
																if(test_bit(e.sym_addr >> PAGE_SHIFT,bitmap))
																	bfs_roots.push(e);
																else
																	printf("page not dirty\n");
															}
														}
													}
												}
											}
										}

									//	if(shadow == 1 && msmt_round > 1)
									//	{
									//		
									//	}
									}
								}
							}
						}
					
					}
					if(detection == 1)
					{
						current_obj.push_back(current_fld);
					}
				}

				//Call verifier function here
				if(detection == 1)
				{
					unique_structs.insert(make_pair(sym_addr,1));
					total_checks++;
					//check_invariants(prefix, current_obj,sym_pathname); // Activate as required, Checks class invariants
					//check_obj_invariants(prefix,current_obj,sym_addr,sym_pathname); // Activate as required, Checks invariants
					current_obj.clear();
				}
			}
			else
			{
#ifdef LOG
				//Log this event and continue
				if(start_logging)
				{
					char logbuffer[MAX_LINE_INPUT_SIZE];
					sprintf(logbuffer,"Invalid page at 0x%x", page_addr);
					logln(logbuffer);
				}
#endif 
			}
		}
		else
		{
			printf("Definition not found ! %s\n",sym_type.c_str());
			//exit(-1);
		}

		//Store node in visited map
		visited.insert(make_pair(sym_addr, 1));	

		//Pop node from the queue
		bfs_roots.pop();
		close_filter_file();

		if(exit_now == 1)
		{
			printf("Stopping dynamic memory scan ... \n");
			return;
		}
	}

	if(bfs_roots.empty())
		cout << "BFS queue empty" << endl;
	else if(total_objects >= MAX_OBJECTS_TRAVERSED)
		cout << "Max objects traversed " << endl;

	//Process linked list nodes
	process_linked_lists(msmt_round);

	print_obj_stats(msmt_round);

	cout << endl << "[" << get_time() << "] End - Dynamic memory scan " << endl;
	
#ifdef INV_LOG
	fprintf(inv_log,"[%s] End - Dynamic memory scan.\n",get_time());			
	fflush(inv_log);
#endif

#ifdef LOG
	if(start_logging)
	{
		char logbuffer[MAX_LINE_INPUT_SIZE];
		sprintf(logbuffer,"Finished dynamic memory scan");
		logln(logbuffer);
	}
#endif
	
}

void process_linked_lists(int msmt_round)
{
	FILE * fd;
	vector<unsigned long long> list_elems;
	if(detection == 0)
	{
		std::stringstream strfile;
		strfile << "snapshots/linkedlist" << msmt_round << ".ir.dtrace";
		fd  = fopen(strfile.str().c_str(),"w");
		if(fd == NULL)
		{
			cout << "Cannot open file " << strfile.str() << endl;	
			exit(-1);
		}
	}

	cout << "ListQ size = " << listq.size() << endl;


	map<unsigned long long, short> heads_seen;
	int skipped_nodes =0;
	int list_count = 0;

	//Print the list of nodes considered as linked list heads
	while(!listq.empty())
	{
		entry_t node = listq.front();
		string actual_type;
		bool root = node.root;

		actual_type = get_container_type(node.sym_type);
		if(strcmp(trim((char *)actual_type.c_str()),"struct list_head") == 0)
		{
			listq.front();
			listq.pop();
			skipped_nodes++;
			continue;
		} 

		unsigned long long sym_addr = node.sym_addr;
		string sym_attr = node.sym_attr;

		map<unsigned long long, short>::iterator hiter = heads_seen.find(sym_addr);

		if(hiter != heads_seen.end())
		{
			//Ignore this node, already processed
			listq.pop();
			continue;
		}		

		heads_seen.insert(make_pair(sym_addr,1));		

		/*fprintf(fd,"%s,%s,%s root=%d\n", node.pathname.c_str(), node.sym_type,node.sym_attr, node.root);
		listq.pop();
		continue; */
		
		//cout << "sym_attr =" << sym_attr << " in_static_area(sym_addr) = " << in_static_area(sym_addr) << endl;
		//Follow the linked list nodes here
		map<unsigned long long, short> seen;
		map<unsigned long long, short>::iterator siter; 
		
		bool close_brace = false;
		char straddr[20];
		if(valid_addr(sym_addr))
		{
			sprintf(straddr,"0x%llx",sym_addr);
			if(detection == 0)
			{
				fprintf(fd, "%s0x%llx:::%s\n[", node.pathname.c_str(),sym_addr, actual_type.c_str());	
			}
			
			close_brace = true;
		}

		int num_nodes = 0;
		unsigned long long start_addr = sym_addr;
		while(valid_addr(sym_addr))
		{
			 /* Find the definition of the node type in definition map */
        	map<string,definition_t>::iterator iter = def_map.find(trim((char *)actual_type.c_str()));

			unsigned long long obj_addr = sym_addr;

			if(!root)
			{
				if((strstr(sym_attr.c_str(),ATTRIBUTE_STRING) != NULL) && (strstr(sym_attr.c_str(),CONTAINER_ATTRIBUTE) != NULL))
				{
					if(iter != def_map.end())
					{
						//Found definition
						vector<field_t> fields = iter->second;

						for(int i=0; i<fields.size(); i++)
						{
							field_t fld = fields[i];
							string fld_type = fld.type;
							string fld_name = fld.name;
							int fld_offset = fld.offset;
							string container_fld_name = get_container_field_name(sym_attr);

							if(strncmp(fld_name.c_str(),container_fld_name.c_str(),strlen(container_fld_name.c_str())) ==0)
							{
								obj_addr = sym_addr - fld_offset;
								break;
							}
						} 
					}
				}
				else
				{
					sym_addr = sym_addr + node.field_offset;
				}
			}

			siter = seen.find(obj_addr);
			if(siter != seen.end())
			{
				//Encountered node already visited
				break;
			}
			else
			{
				num_nodes++;
				seen.insert(make_pair(obj_addr,1));
			}

			//Print address of this node
			if(!root)
			{
			if(detection == 0) 
			{
				fprintf(fd, " %lld ", obj_addr);	
			}
			else
			{
				list_elems.push_back(obj_addr);
			}
			}
			root = false;
			//fprintf(fd, "0x%llx ", obj_addr);	
			
			//Get next node
			unsigned long long page_addr = page_addr_from_sym_addr(sym_addr);
			if(valid_page(page_addr)) 
            {
            	page_t new_page = get_page(page_addr,msmt_round);
            	int offset = sym_addr - page_addr;
				sym_addr = *(unsigned int *)(new_page.page_data + offset);

				if(sym_addr == 0)
				{
					break;
				}

				//Make sure that the next link is not pointing again to the list head
				if(sym_addr == start_addr)
				{
					break;
				} 
            }                                                                                                                                  
			else
			{
				break;
			}
		}
		if(close_brace)
		{
			if(detection == 0)
			{
				fprintf(fd, "]\n");	
			}
			else
			{

				stringstream tmpstr;
				tmpstr << straddr;	

				string elem_name = node.pathname; 
				elem_name.append(straddr);
				elem_name.append("[]");
				listvals.insert(make_pair(elem_name, list_elems));
				
			}
		}

		list_elems.clear();
		seen.clear();
		listq.pop();
	}
	heads_seen.clear();

	if(detection == 1)
	{
		//cout <<  "Size of listvals " << listvals.size() << endl;

		map<string,vector<unsigned long long> >::iterator iter = listvals.begin();

		while(iter != listvals.end())
		{
			//fprintf(inv_alerts,"{%s}\n", (iter->first).c_str());
			++iter;
		}

		check_seq_invariants();
		listvals.clear();
	}
	else
	{
		fclose(fd);
	}
}

void print_obj_stats(int msmt_round)
{
	FILE * stat;
 	std::stringstream strtmp;
    strtmp<< msmt_round;
                                                                                                                                               
    string filename = "snapshots/stat_objs" + strtmp.str() + ".txt";

	stat = fopen(filename.c_str(),"w");
	if(stat == NULL)
	{
		cout << "Cannot open file " << filename << endl;
		exit(-1);
	}

	map<string,int>:: iterator ts_iter = type_stats.begin();
	int total_obj_count =0;
	while(ts_iter != type_stats.end())
	{
		fprintf(stat, "%s %d\n", (ts_iter->first).c_str(), ts_iter->second);
		total_obj_count += ts_iter->second;
		ts_iter++;
	}
	fprintf(stat, "Total Objs = %d\n", total_obj_count);
	fclose(stat);
	type_stats.clear();
	
}

string get_container_type(string strtype)
{
	char strobjtype[MAX_LINE_INPUT_SIZE];
	char * obj_type;

	strcpy(strobjtype, strtype.c_str());
	if(strstr(strobjtype,ATTRIBUTE_STRING) != NULL)
	{
		//Get the object type from within the container field
		obj_type = strstr(strobjtype, "sizeof("); 
		obj_type = obj_type + 7;

		char * tmp = strstr(strobjtype, "*))))");
		tmp[0]='\0';
		
		//Get rid of the field name of container struct
		char * ptr = strstr(obj_type,"___");
		if(ptr != NULL)
		{
			ptr[0]='\0';
		}	

		ptr = strstr(obj_type," const");
		if(ptr != NULL)
		{
			ptr[0]='\0';
		}
	}
	else
	{
		//The first two words give the object type
		for(int k=0;k < strlen(strobjtype); k++)
		{
			if(strobjtype[k] == '*')
			{
				strobjtype[k] = '\0';
				break;
			}
		}
		obj_type = strobjtype;
	}

	string ret = obj_type;

	return(ret);
}

string get_container_field_name(string strtype)
{
    char strobjtype[MAX_LINE_INPUT_SIZE];
    char * obj_type;
	string ret;
                                                                                                                                               
    strcpy(strobjtype, strtype.c_str());
    if(strstr(strobjtype,ATTRIBUTE_STRING) != NULL)
    {
        //Get the object type from within the container field
        obj_type = strstr(strobjtype, "sizeof(");
        obj_type = obj_type + 7;
                                                                                                                                               
        char * tmp = strstr(strobjtype, "*))))");
        tmp[0]='\0';

		//Extract the field name
    	char * ptr = strstr(obj_type,"___");
		if(ptr != NULL)
		{
			ptr = trim(ptr);
			ptr = ptr+3;
    		ret = ptr;
		}
		else
		{
			ret="";
		}
    }
    else
    {
        //The first two words give the object type
        /*for(int k=0;k < strlen(strobjtype); k++)
        {
            if(strobjtype[k] == '*')
            {
                strobjtype[k] = '\0';
                break;
            }
        }
        obj_type = strobjtype; */
		ret ="";
    }
                                                                                                                                               

                                                                                                                                               
                                                                                                                                               
    return(ret);
}


bool ptr_to_composite_type(string strtype)
{
	const char * str = strtype.c_str();

	//if(strstr(str,"(*)(") != NULL)
	if(func_ptr((char *)str))
	{
		return(false);
	}	
	else if(strstr(str,"**") != NULL)
	{
		return(false);
	} 
	else if(strstr(str,"*") != NULL)
	{
		if((strstr(str,"struct") != NULL) || (strstr(str,"union") != NULL))
		{
			return(true);
		}
	}
	return(false);
}

extern FILE * fd_log;
page_t get_page(unsigned long long page_addr, int msmt_round)
{
	assert((valid_addr(page_addr)));
#ifdef LOG
	char logbuffer[MAX_LINE_INPUT_SIZE];
	sprintf(logbuffer,"Fetching page: 0x%llx\n", page_addr);
	log(logbuffer);
	fflush(fd_log);
#endif
	unsigned long long orig_addr = page_addr;
	
	if(page_addr >=PAGE_OFFSET)
	{	
		//JEFF CHANGED for XenAccess conversion
		//printf("Convert virt to phys\n");
		//printf("Xen translated %d\n",xa_translate_kv2p(&xai,page_addr));  	
		page_addr = virt_to_phys(page_addr);
		//page_addr = xa_translate_kv2p(&xai,page_addr);
		//printf("Her translated %d\n",page_addr);

		/*if(page_addr != 0 && live_p2m[page_addr >> PAGE_SHIFT] == 0)
		{
			// if in dynamic_pfn_map remove

			multimap<unsigned long long, entry_t>::iterator iter = dynamic_pfn_map.find(page_addr >> PAGE_SHIFT);
			if(iter != dynamic_pfn_map.end())
			{
				pair<multimap<unsigned long long,entry_t>::iterator,multimap<unsigned long long,entry_t>::iterator> entries;
				entries = dynamic_pfn_map.equal_range(page_addr >> PAGE_SHIFT);

				multimap<unsigned long long, entry_t>::iterator itr;

				for (itr=entries.first; itr!=entries.second; ++itr)
				{
					if(itr->first == orig_addr)
					{
						dynamic_pfn_map.erase(itr);
						printf("remove entry\n");
					}
				}				

				//dynamic_pfn_map.erase(entries.first,entries.second);
			}
		}*/
			
	}
	
	page_t new_page;
	//Getting rid of the page map to save memory
	map<unsigned long long, page_t>::iterator iter = page_map.find(page_addr);

	if(iter != page_map.end())
  	{
    		//Page exists
		new_page =  iter->second;
	}
	else
	{ 
		//Page does not exist, fetch the page and store page in page_map for future accesses
		//printf("Fetching page 0x%x\n",page_addr);
		fetch_page(page_addr,0,msmt_round);

		//Page now in dmadata_buffer
		new_page.page_begin = page_addr;

		copy_bytes(dmadata_buffer, new_page.page_data, SLAB_SIZE,0);

		//Stores page for future accesses in current scan
		if(page_map.size() < 100000)
		{
			page_map.insert(make_pair(page_addr,new_page));
		}
	} 
	return(new_page);
}

void bfsnodes_from_initroots(queue<entry_t> &bfs_roots)
{
	//First handle pointers from the init_roots set.
	for(int i=0; i<init_roots.size(); i++)
	{

		int root_index = init_roots[i];
		//Get the value of the current pointer from static_symbol
		unsigned char val_addr[4];
		int last_entry = static_symbol[root_index].values.size();
		unsigned int obj_addr;
		const char * obj_type;
		char strobjtype[MAX_LINE_INPUT_SIZE];


		//printf("bfsnodes_from_init: %s %s\n",static_symbol[root_index].sym_name,static_symbol[root_index].sym_type);
	
		assert(static_symbol[root_index].values.size() > 0);
		map<unsigned long long, char>::iterator iter = visited.find(static_symbol[root_index].sym_addr);
		if(iter == visited.end())
		{
			//Mark the static symbol addr as visited 
			visited.insert(make_pair(static_symbol[root_index].sym_addr, 1));		
		}
                                                                                                                                               
        for(int k=0; k<4; k++)
        {
       		val_addr[k] = static_symbol[root_index].values[k];
        } 

		obj_addr = *(unsigned int *)(val_addr);
#ifdef LOG
		if(start_logging)
		{	
			char logbuffer[MAX_LINE_INPUT_SIZE];
			sprintf(logbuffer,"Looking at Root = %s Value=0x%x", static_symbol[root_index].sym_name, obj_addr);
			logln(logbuffer);
		}
#endif

		if((obj_addr == static_symbol[root_index].sym_addr) || (obj_addr == (static_symbol[root_index].sym_addr - 4))) 
		{
			//Anyway, add it to the visited list
			map<unsigned long long, char>::iterator iter = visited.find(obj_addr);
			if(iter == visited.end())
			{
				//Mark node as visited
				visited.insert(make_pair(obj_addr, 1));		
			}
			continue;	
		}
		
		if(valid_addr(obj_addr))
		{
			map<unsigned long long, char>::iterator iter = visited.find(obj_addr);

			//If node not already in visited list
			if(iter == visited.end())
			{
				//Mark node as visited
				visited.insert(make_pair(obj_addr, 1));		

				//Also mark the original static addr as visited
				visited.insert(make_pair(static_symbol[root_index].sym_addr,1));
				string container_type = get_container_type(static_symbol[root_index].sym_type);
				obj_type = container_type.c_str();
#ifdef LOG
				if(start_logging)
				{
					char logbuffer[MAX_LINE_INPUT_SIZE];
					sprintf(logbuffer,"Root = %s Value=0x%x Type=%s", static_symbol[root_index].sym_name, obj_addr, obj_type);
					logln(logbuffer);
				}
#endif

				//Create a new entry_t for this object
				entry_t ent;

				ent.sym_addr = obj_addr;	
				ent.sym_id = obj_addr;
				ent.sym_attr[0] ='\0'; ;
				ent.pathname = static_symbol[root_index].sym_name;
				ent.pathtype= static_symbol[root_index].sym_type;
				
				// JEFF PATHADDR
				std::stringstream pathout;
				pathout << "0x" << hex << static_symbol[i].sym_addr;
				ent.pathaddr.push_back(pathout.str());


				if((strstr(static_symbol[root_index].sym_type,ATTRIBUTE_STRING) != NULL) &&  
					(strstr(static_symbol[root_index].sym_type,CONTAINER_ATTRIBUTE) != NULL))
				{
					strcpy(ent.sym_attr,static_symbol[root_index].sym_type);
				}
				strcpy(ent.sym_type, obj_type); //C410 - Moved below


				assert(valid_addr(ent.sym_addr));

//#ifdef WHITE_LIST
		//		if(in_white_list(ent.sym_type))
//#else
				if(!in_black_list(ent.sym_type))
//#endif											
				{

			//		printf("Pushing in BFS ROOTS object, addr=0x%llx, type=%s, attr=%s\n",ent.sym_addr,  ent.sym_type, ent.sym_attr);
#ifdef LOG 
					if(start_logging)
					{	
						char logbuffer[MAX_LINE_INPUT_SIZE];
						sprintf(logbuffer,"Pushing in BFS ROOTS object, addr=0x%llx, type=%s, attr=%s",ent.sym_addr,  ent.sym_type, ent.sym_attr);
						logln(logbuffer);
					}
#endif
					if(valid_addr(ent.sym_addr))
					{
						//if(!in_static_area(ent.sym_addr)) //C45
						{
							assert((ent.sym_addr == 0x000000) ||valid_addr(ent.sym_addr));
							if(!in_startset(root_index))
							{
								ent.root = true;
							}
							else
							{
								ent.root = false;
							}
#ifdef LOG
							char logbuffer[MAX_LINE_INPUT_SIZE];
							sprintf(logbuffer,"0x%llx %s %s root=%d",static_symbol[root_index].sym_addr, static_symbol[root_index].sym_name, static_symbol[root_index].sym_type, ent.root);
							logln(logbuffer);
#endif
							bfs_roots.push(ent);
						}
					}
				}
				else
				{	
				//	printf("Skipping BFS ROOTS object in black list, addr=0x%llx, type=%s, attr=%s\n",ent.sym_addr,  ent.sym_type, ent.sym_attr);
#ifdef LOG 			
					if(start_logging)
					{
						char logbuffer[MAX_LINE_INPUT_SIZE];
						sprintf(logbuffer,"Skipping BFS ROOTS object in black list, addr=0x%llx, type=%s, attr=%s",ent.sym_addr,  ent.sym_type, ent.sym_attr);
						logln(logbuffer);
					}
#endif
				}
			}
		}
	}
	cout << "bfsnodesfrom_init roots: num nodes after= "<< listq.size() << endl;
}



void fetch_page(unsigned int current_mem_start, int block_no, int msmt_round )
{

	uint32_t addr,offset;
	unsigned char *memory = NULL;

	//printf("fetching page 0x%x\n",current_mem_start);

	addr = (uint32_t) current_mem_start;

	// JEFF added check for address 0
	if(addr != 0){

		if(addr != 0xffffffff && addr != 0x55555555 && addr >> PAGE_SHIFT < p2m_size)
		{		
			uint32_t pfn = addr >> PAGE_SHIFT;
			uint32_t mfn = live_p2m[pfn];
		//	printf("pfn: 0x%x mfn:0x%x\n",pfn,live_p2m[pfn]);	
			if(mfn != 0xffffffff && mfn != 0)
			{			
				memory = memory_map + (pfn * 0x1000);

				//memory = (unsigned char *)xa_access_ma(&xai, mfn << PAGE_SHIFT, &offset, PROT_READ);
			
				if(memory == NULL){
					printf("failed to map memory\n");
					//exit(-1);
				}
		
				copy_bytes(memory,dmadata_buffer,xai.page_size, 0);
				//munmap(memory,PAGE_SIZE);
			}
		}
		/* for shared info page - not in pfn table*/
		/*else
		{
			printf("HERE\n");
			uint32_t mach = xa_translate_kv2p(&xai,virt_addr);
			printf("mach after xa_translate: 0x%x\n",mach);
			
			if(mach != 0)
			{		
				unsigned char *xa_memory = (unsigned char *)xa_access_ma(&xai, mach, &offset, PROT_READ);
				copy_bytes(xa_memory,dmadata_buffer,xai.page_size, 0);
				munmap(xa_memory,PAGE_SIZE);
			}
		}*/
	}

}

unsigned long long page_addr_from_sym_addr(unsigned long long sym_addr)
{
	unsigned long long page_addr = sym_addr / SLAB_SIZE;
    page_addr = page_addr * SLAB_SIZE;

	return(page_addr);
}


/*

printf("addr: 0x%x ",addr);
			printf("xa_translated: 0x%x\n",mach);
		//	printf("mach: 0x%x\n",mach);
			uint32_t mfn = mach >> PAGE_SHIFT;
		//	printf("mfn: 0x%x\n",mfn);
			uint32_t pfn = live_m2p[mfn];
		//	printf("pfn: 0x%x\n",pfn);
			if(pfn == 0xffffffff || pfn == 0x55555555)
				return pfn;
			else
				return pfn << PAGE_SHIFT;	
*/


unsigned long long pt_walk(unsigned long long vaddr)
{
		//uint32_t mach = xa_translate_kv2p(&xai,vaddr);
	//	printf("vaddr: 0x%x ",vaddr);
	//	printf("xa_translated: 0x%x\n",mach);

		uint32_t cr3 = xai.cr3;

	//	printf("cr3: 0x%x\n",cr3); 
	//	printf("pae: %d\n",&xai.pae);

		/* GET PDPTE3 VALUE */
        /* page directory pointer table */
        uint32_t pdptb = cr3 & 0xFFFFFFE0;
        uint32_t pdpi_index = (vaddr >> 30) * sizeof(uint64_t);
        uint32_t pdpi_entry = pdptb + pdpi_index;  // IN INTEL PDPTEi
        uint64_t pdpie_value;
		memcpy(&pdpie_value,(void *)(memory_map+(live_m2p[pdpi_entry >> PAGE_SHIFT] * 0x1000)+((pdpi_entry << 20) >> 20)),sizeof(uint64_t));

        /* GET PDE VALUE */
        uint64_t pde_base = pdpie_value & 0xFFFFFF000ULL;
        uint32_t pde_index = (((vaddr) >> 21) & 0x1FF) * sizeof(uint64_t);
        uint32_t pde_entry = pde_base + pde_index;  // IN INTEL PDE
		//printf("pde_entry: 0x%x 0x%x\n",pde_entry,(pde_entry << 20) >> 20);       
		uint64_t pde_value;
		memcpy(&pde_value,(void *)(memory_map+(live_m2p[pde_entry >> PAGE_SHIFT] * 0x1000)+((pde_entry << 20) >> 20)),sizeof(uint64_t));
		//printf("pde_value: 0x%x\n",pde_value);

		//if(pde_value == 0)
			return 0;
		//else
		//{
//
		    /* GET PTE VALUE */
		 /*   uint64_t pte_base = pde_value & 0xFFFFFF000ULL;
		    uint64_t pte_index = (((vaddr) >> 12) & 0x1FF) * sizeof(uint64_t);
		    uint32_t pte_entry = pte_base + pte_index;  // IN INTEL PTE
			uint64_t pte_value;
			memcpy(&pte_value,(void *)(memory_map+(live_m2p[pte_entry >> PAGE_SHIFT] * 0x1000)+((pte_entry << 20) >> 20)),sizeof(uint64_t));
			printf("pte_value: 0x%x\n",pte_value);

			uint32_t page = pte_value & 0xFFFFFF000ULL | (vaddr & 0xFFF);
		    printf("page: 0x%.8x\n",page);*/

		//	return xa_translate_kv2p(&xai,vaddr);
		//}
}


unsigned long long virt_to_pfn(unsigned long long addr)
{
//	printf("convert: 0x%x\n",addr);
//	printf("normal translate: 0x%x\n",(addr-PAGE_OFFSET)>>PAGE_SHIFT);
//	printf("xa translate: 0x%x\n",xa_translate_kv2p(&xai,addr));

	if(addr < VMALLOC_START)
		//return live_m2p[xa_translate_kv2p(&xai,addr) >> PAGE_SHIFT];
		return((addr - PAGE_OFFSET) >> PAGE_SHIFT);
	else
	{
	//	printf("greater than VMALLOC_START\n");
	//	uint32_t mach = xa_translate_kv2p(&xai,addr);
	//	printf("mach: 0x%x\n",mach);
	//	uint32_t mfn = mach >> PAGE_SHIFT;
	//	printf("mfn: 0x%x\n",mfn);
	//	uint32_t pfn = live_m2p[mfn];
	 //	printf("pfn: 0x%x\n",pfn);
	//	return pfn;
		return pt_walk(addr) >> PAGE_SHIFT;
	}
}

unsigned long long virt_to_phys(unsigned long long addr)
{	
	translate = 1;
//	printf("translate: 0x%x\n",addr);
	virt_addr = addr;
//printf("test\n");
	//printf("offset translate: 0x%x\n", live_p2m[(addr-PAGE_OFFSET) >> PAGE_SHIFT]<<PAGE_SHIFT);
//	printf("xa_translate: 0x%x\n",xa_translate_kv2p(&xai,addr));

	if(addr != 0x0)
	{
		if(addr < VMALLOC_START)
			return((addr - PAGE_OFFSET));
		//	return live_m2p[xa_translate_kv2p(&xai,addr) >> PAGE_SHIFT] << PAGE_SHIFT;
		else
		{
			//printf("greater than VMALLOC_START\n");
			return pt_walk(addr);
		}
		//return(xa_translate_kv2p(&xai,addr));
		//return(addr);
	}
	return(addr);
}

bool in_reserved_space(unsigned long long addr)
{
	if((addr >= 0xc00a0000) && (addr <= 0xc00fffff))
		return true;
	else
		return false;
}

bool valid_page(unsigned long long page_addr)
{
	//if((page_addr == 0x0) || (page_addr == 0xffffffff))
	if((page_addr == 0x0) || (page_addr >= 0xf8000000) || (page_addr <=0xc0000000) ||  in_reserved_space(page_addr))
		return false;
	else
		return true;
}

bool valid_addr(unsigned long long sym_addr)
{
	if((sym_addr > 0xc0000000) && (sym_addr < 0xf8000000) && (!in_reserved_space(sym_addr)))
	{
		return(true);
	}
	else
	{
		return(false);
	}
}

void remove_const_keyword(string &sym_type)
{
	char * ptr;

	ptr =(char *) strstr(sym_type.c_str()," const");
	if(ptr != NULL)
	{
		ptr[0]='\0';
	}

	//JEFF remove const from front too
	ptr = (char *)strstr(sym_type.c_str(),"const");
	if(ptr != NULL)
	{
		sym_type.erase(0,6);
	}
}

bool in_white_list(char *str)
{
	vector<string> white_list;
	string type = trim(str);

	white_list.push_back("struct task_struct");
	//white_list.push_back("struct list_head");

	for(int i=0; i<white_list.size(); i++)
	{
		if(white_list[i] == type)
		{
			return(true);
		}
	}
	return(false);
}

bool in_black_list(char * str)
{
	vector<string> black_list;
	string type = trim(str);	

	/*black_list.push_back("struct tty_struct");
	black_list.push_back("struct sock");
	black_list.push_back("struct tq_struct");
	black_list.push_back("struct vfsmount"); 
	black_list.push_back("struct page"); */
	//black_list.push_back("struct buffer_head"); 
//T	black_list.push_back("struct dentry"); 
	black_list.push_back("bio_destructor_t");
//T	black_list.push_back("struct map_segment");
//T	black_list.push_back("struct idr_layer");
	black_list.push_back("struct mtd_info");
//T	black_list.push_back("struct hotplug_slot");	
//	black_list.push_back("struct file");
//T	black_list.push_back("struct xt_table");
//	black_list.push_back("struct macvlan_port");
//	black_list.push_back("struct garp_port");
//T	black_list.push_back("struct msi_desc");
/*	black_list.push_back("struct wireless_dev");
	black_list.push_back("struct ieee80211_device");
	black_list.push_back("struct proc_dir_entry");		
	black_list.push_back("struct dentry");
	black_list.push_back("struct module");
	black_list.push_back("struct nsproxy");
	black_list.push_back("struct cgroupfs_root");
*/
	for(int i=0; i<black_list.size(); i++)
	{
		if(black_list[i] == type)
		{
			return(true);
		}
	}
	return(false);
}

bool ll_head(char * sym_type)
{
	if(((strstr(sym_type,"list_head") != NULL) && (strstr(sym_type,CONTAINER_ATTRIBUTE) != NULL)) || 
		(strstr(sym_type,"next") != NULL) || (strstr(sym_type,"prev") != NULL))
	{
		return(true);
	}
	/*else if((strstr(ent.sym_attr,"list_head") != NULL) && (strstr(ent.sym_attr,CONTAINER_ATTRIBUTE) != NULL))
	{
		return(true);
	}*/
	else
	{
		return(false);
	}
}

#define STATIC_AREA_BEGIN 0xc0285680
#define STATIC_AREA_END 0xc032c0c0

bool in_static_area(unsigned long long ptr)
{
	if((ptr >=STATIC_AREA_BEGIN) && (ptr <= STATIC_AREA_END))
	{
		return(true);
	}
	return(false);
}

bool list_head(char * type)
{
	if(strstr(type, "list_head") != NULL)
	{
		return(true);
	}
	return(false);
}
/*
void my_send_callback (struct gm_port *port, void *context, gm_status_t status)
{
	count_sends++;

	if (status != GM_SUCCESS)
	{
			if (status != GM_SEND_DROPPED)
			{
					gm_perror ("send completed with error", status);
			}
	}
}
*/

/*void log_assert(char * str)
{
	char logbuffer[MAX_LINE_INPUT_SIZE];

	sprintf(logbuffer,"[Assert failed at %d] ",__LINE__);
	log(logbuffer);

	strcpy(logbuffer,str);
	logln(logbuffer);
}*/

