/*---------------------------------------------------------------------------
 * File:	nfa_bdd_simulate.cc
 *
 *
 * Author:	Rezwana Karim
 * Date:	august 24, 2009
 *
 * Match test for NFA_BDD implementations
 * Switch to measure the performance by CPU cycles
 *------------------------------------------------------------------------
 *
 *
 *
 *-----------------------------------------------------------------------*/


#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>

#include "nfa_operation.h"
#include "clock.h"
#include "packet.h"
#include "decode.h"


#define BUF_SIZE		200000

//#define DEBUG

/* Choose ONE implementation when performing simulation */

#define NFA_IMPLEMENTED_BY_BDD    

using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;


string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);

int main(int argc, char** argv) {

    unsigned long long starttm = 0;
    unsigned long long stoptm = 0;
    unsigned long long total_cycles = 0;
    size_t total_bytes = 0, total_payload_bytes = 0;
    unsigned int t_start, t_end, num_filled = 0;



 #ifdef NFA_IMPLEMENTED_BY_BDD
     NFAOperationByBDD nfa_bdd_operate_obj;
 #endif

    if (argc != 5) {
        cerr << "Usage:" <<  argv[0]<<"  <trace file> <y indicating Acceptance signal> <path to BDD directory> <number of trans chunk>" << endl;
        exit(1);
    }
    
    /*BDD based NFA is already constructed using a separate process , now simulate nfa operation*/
        
    nfa_bdd_operate_obj.readBddInfoFromFile(argv[3], argv[4]);
    /* initilize the NFA expressed by BDD and also set the mapping*/
    nfa_bdd_operate_obj.initNFA();
    /*todo : merge this into initNFA*/
    nfa_bdd_operate_obj.formCubeForAbstract();
        
    bool check_detect;
    
    if((argv[2]!=NULL)&&(strcmp(argv[2],"y")==0))
      check_detect=true;
    else
      check_detect=false;
       
        
    /* ----read stream for match test---- */
    trace_file = argv[1];

    init_pcap(&pcap_handle, trace_file);
    /*track starting time before starting nfa operation*/
    t_start = cputime();

    raw_packet_t rp;
    int i = 0;
    unsigned char *iphdr, *tcphdr;
    size_t tcphdr_len = 20;	/* default size of tcp hdr */
    size_t iphdr_len = 20;
    int payload_sz = 0;
    unsigned char *payload;
    while (get_next_packet(pcap_handle, &rp)) {
        iphdr = rp.pkbuf + 14;	/* skip the link hdr */
        iphdr_len = (*iphdr & 0x0f) * 4; /* extract the length of ip hdr */
        tcphdr = iphdr + iphdr_len;
        tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
        payload = tcphdr + tcphdr_len;
        payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;
	if (payload_sz < 0)
	  cout << "negative payload size" << endl;
	if (payload_sz > 0) {  
          cout << "The " << i << "th packet len" << payload_sz << endl;          
          rdtsc(starttm);
          #ifdef NFA_IMPLEMENTED_BY_BDD
            nfa_bdd_operate_obj.simulate(payload, payload_sz, check_detect);
          #endif
          rdtsc(stoptm);
          total_payload_bytes += payload_sz;
          total_cycles += (stoptm - starttm);
          if (rp.allocated) {
            free(rp.pkbuf);
            rp.allocated = 0;
          }
	}
        total_bytes += rp.hdr.caplen;
 	i++;
	//cout << "The " << i << "th packet" << endl;
    }

    t_end = cputime();
    cout << "Memory in use: " << Cudd_ReadMemoryInUse(nfa_bdd_operate_obj.manager) << endl;   
    /*free memory*/
    nfa_bdd_operate_obj.freeMemory();
    cout << "Total payload bytes: " << total_payload_bytes << endl;
    cout << "Total bytes including headers: " << total_bytes << endl;
    cout << "BDD based NFA Simulation: execution time is " << (double)total_cycles/(double)total_payload_bytes << " cycles/byte" << endl;

    cout << "Total time spent on simulation: " << (double)(t_end - t_start)/(double)1000 << " seconds" << endl;
      
  
    return 0;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

