/**------------------------------------------------------
 * File:     nfa_operation.cpp
 * Author :  Rezwana Karim
 * Date:     August 24, 2009
 *--------------------------------------------------------------------------
 * Simulates the NFA operation on BDD using NFAOperationByBDD class 
 *-------------------------------------------------------------------------- 
 **/


#include <iostream>
#include <fstream>
#include <cstdlib>

#include <cmath>
#include <vector>


using namespace std;
/*Set these file paths appropriately based on where the CUDD package lies*/

#include "../../cudd-2.4.2/include/cuddObj.hh"
#include "../../cudd-2.4.2/include/cuddInt.h"
#include "../../cudd-2.4.2/dddmp/dddmpInt.h"



#include "nfa_operation.h"


#define MAX_INPUT_LENGTH 20000
#define NO_OF_TEST_RUNS 10

//#define DEBUG
//#define ACCEPT
#define CHECK_CHUNK_LOAD
#define BOTTLE


int acc_duration;
int detection_count;
int NO_OF_BDD_FILES;	
char bdd_src_dir[300];

Dddmp_RootMatchType rootmatchmode;
Dddmp_VarMatchType varmatchmode;
Dddmp_VarInfoType varoutinfo;


void NFAOperationByBDD::readBddInfoFromFile(char *bdd_src_dir_path, char* trans_chunk_num){
           
    strcpy( bdd_src_dir, bdd_src_dir_path);
    NO_OF_BDD_FILES = atoi( trans_chunk_num);	
    char infoFile[500];
    
    int len = strlen(bdd_src_dir); 
    strcpy(infoFile, bdd_src_dir);
    strcpy( infoFile + len,"/metadata/nfa_info.txt");
    
    ifstream inputFile;
    inputFile.open( infoFile );
    if(!inputFile){
        cout<<"nfainfo file not found"<<endl;
        exit(1);
    }
    inputFile>>STATE_ENCODE_SIZE;
    inputFile>>INPUT_ENCODE_SIZE;
    inputFile>>NO_OF_PAIRS;
    
    signatureIDArray= new unsigned short int [NO_OF_PAIRS];
    unsigned int i;
    for(i=0;i<NO_OF_PAIRS;i++){
        inputFile>>signatureIDArray[i];
    }
    inputFile.close();
    cout<<STATE_ENCODE_SIZE<<" "<<INPUT_ENCODE_SIZE<<endl;
}


/**
 * initialize the NFA expressed by BDD
 * initialize
 *
 *     *transFunc
 *     *startSetOfStates
 *     *acceptingSetOfstates
 *     *currentSetOfStates;
 *     setMapping
 **/
void NFAOperationByBDD::initNFA(){
    
    
    /*init dddmp*/ 
    rootmatchmode = DDDMP_ROOT_MATCHLIST;
  #if 1
    varmatchmode = DDDMP_VAR_MATCHIDS;
  #else
    varmatchmode = DDDMP_VAR_MATCHNAMES;
  #endif
    varoutinfo = DDDMP_VARIDS;         
    
    /*set the number of variables*/
    
    int numberOfVars=2*STATE_ENCODE_SIZE+INPUT_ENCODE_SIZE;
     
    /*----------------------- Init Var Information Structure ------------------*/

    varInfo.nDdVars = numberOfVars;

    varInfo.rootNames = NULL;
    varInfo.ddType = DDDMP_NONE;
    varInfo.nVars = (-1);
    varInfo.nSuppVars = (-1);
    varInfo.varNamesFlagUpdate = 1;
    varInfo.suppVarNames = NULL;
    varInfo.orderedVarNames = NULL;
    varInfo.varIdsFlagUpdate = 1;
    varInfo.varIds = NULL;
    varInfo.varIdsAll = NULL;
    varInfo.varComposeIdsFlagUpdate = 1;
    varInfo.varComposeIds = NULL;
    varInfo.varComposeIdsAll = NULL;
    varInfo.varAuxIdsFlagUpdate = 1;
    varInfo.varAuxIds = NULL;
    varInfo.varAuxIdsAll = NULL;
    varInfo.nRoots = (-1);

    /** Initialize the manager
     * numberOfVars     : Initial number of BDD variables
     * 0                : Initial number of ZDD variables
     * CUDD_UNIQUE_SLOTS: Initial size of the unique tables (256)
     * CUDD_CACHE_SLOTS : Initial size of the cache (262144)
     * 0                : target maximum memory (0 means unlimited)
     **/ 
   
    manager = Cudd_Init(numberOfVars, 0, CUDD_UNIQUE_SLOTS, CUDD_CACHE_SLOTS, 0);
   
    Cudd_SetMinHit(manager, 15);
    currentStateVar= new DdNode* [STATE_ENCODE_SIZE];
    nextStateVar=new DdNode* [STATE_ENCODE_SIZE];
    inputVar=new DdNode*[INPUT_ENCODE_SIZE];
    
    
    unsigned short int i;
    
    for(i=0;i<STATE_ENCODE_SIZE;i++){
        
        currentStateVar[i]=Cudd_bddIthVar(manager, INPUT_ENCODE_SIZE + STATE_ENCODE_SIZE-1-i);
        nextStateVar[i]=Cudd_bddIthVar(manager, STATE_ENCODE_SIZE+INPUT_ENCODE_SIZE+STATE_ENCODE_SIZE-1-i);
    }
    
    for(i=0;i<INPUT_ENCODE_SIZE;i++)
        inputVar[i]=Cudd_bddIthVar(manager, INPUT_ENCODE_SIZE-1-i);
    
    /*set the mapping*/
    Cudd_SetVarMap(manager, nextStateVar, currentStateVar, STATE_ENCODE_SIZE);
    
    /*initializing transFunc to 0*/
    transFunc=Cudd_ReadLogicZero(manager);
    /*load bdd's for transition  function and combine them*/
    DdNode **transitionChunksBDD;//=new DdNode*[3];
    transitionChunksBDD=new DdNode* [NO_OF_BDD_FILES];
    /*get fileName*/
   char *transFileName=new char[300];
   DdNode *temp;
   #ifdef CHECK_CHUNK_LOAD
    double minterm_count=0;
   #endif
    
    for(i=0;i<NO_OF_BDD_FILES;i++){
        
        getFileName(i+1, transFileName);
        
         cout<< " chunk file name is "<< transFileName<< endl;

        transitionChunksBDD[i] = Dddmp_cuddBddLoad(manager, varmatchmode, varInfo.orderedVarNames,
            varInfo.varIdsAll, varInfo.varComposeIdsAll, DDDMP_MODE_DEFAULT,
            transFileName, NULL);
        #ifdef CHECK_CHUNK_LOAD
            minterm_count+=Cudd_CountMinterm(manager, transitionChunksBDD[i], numberOfVars);
        #endif
        
        temp=Cudd_bddOr(manager, transFunc, transitionChunksBDD[i]);
        Cudd_Ref(temp);
        Cudd_RecursiveDeref(manager, transFunc);
        transFunc=temp;
    }
    
    #ifdef CHECK_CHUNK_LOAD
            if(minterm_count==Cudd_CountMinterm(manager, transFunc, numberOfVars))
                cout<<"combination successful"<<endl;
            else
                cout<<"combination problem"<<endl;
    #endif
   unsigned long int nodes_num=Cudd_DagSize(transFunc);
    cout<<"number of nodes in transFunc "<< nodes_num << " leaves "<<Cudd_CountLeaves(transFunc)<<endl;    
    
     int len = strlen(bdd_src_dir); 
     
    /*loading start set of states*/
    char startFileName[500];  
    strcpy( startFileName, bdd_src_dir);
    strcpy( startFileName + len,"/metadata/start.txt");
    
    cout << "start "<< startFileName <<endl;
    startSetOfStates = Dddmp_cuddBddLoad(manager, varmatchmode, varInfo.orderedVarNames,
            varInfo.varIdsAll, varInfo.varComposeIdsAll, DDDMP_MODE_DEFAULT,
            startFileName, NULL);
    
    /*loading start set of states*/
    char acceptFileName[500];  
    strcpy( acceptFileName, bdd_src_dir);
    strcpy( acceptFileName + len,"/metadata/accept.txt");
    cout << " accept "<< acceptFileName;
    acceptSetOfStates = Dddmp_cuddBddLoad(manager, varmatchmode, varInfo.orderedVarNames,
            varInfo.varIdsAll, varInfo.varComposeIdsAll, DDDMP_MODE_DEFAULT,
            acceptFileName, NULL);
    
    
    currentSetOfStates=startSetOfStates;
    
        
    /*loading accepting state bdds*/
    varInfo.nDdVars=STATE_ENCODE_SIZE;
    
    char acceptArrayFileName[500];  
    strcpy( acceptArrayFileName, bdd_src_dir);
    strcpy( acceptArrayFileName + len,"/metadata/accept_bdd_array.txt");
        
    cout << "accept array" << acceptArrayFileName << endl;
    int nRoots= Dddmp_cuddBddArrayLoad(manager, rootmatchmode,
                varInfo.rootNames, varmatchmode,
                varInfo.orderedVarNames, varInfo.varIdsAll, varInfo.varComposeIdsAll,
                DDDMP_MODE_DEFAULT, acceptArrayFileName, NULL, &acceptStateBddArray);
    
    #ifdef ACCEPT
     for(i=0;i<NO_OF_PAIRS;i++)
         printBDD(acceptStateBddArray[i],STATE_ENCODE_SIZE);
    #endif
        
    
    /*loading input symbol bdds*/
    varInfo.nDdVars=INPUT_ENCODE_SIZE;
    
    char inputSymbolFileName[500];  
    strcpy( inputSymbolFileName, bdd_src_dir);
    strcpy( inputSymbolFileName + len,"/metadata/input_bdd_array.txt");
    
    cout << "input "<< inputSymbolFileName << endl;
    nRoots= Dddmp_cuddBddArrayLoad(manager, rootmatchmode,
                varInfo.rootNames, varmatchmode,
                varInfo.orderedVarNames, varInfo.varIdsAll, varInfo.varComposeIdsAll,
                DDDMP_MODE_DEFAULT, inputSymbolFileName, NULL, &inputSymbolBddArray);
    
}



/**
 * form cube for abstract after existing quantification
 **/

void NFAOperationByBDD::formCubeForAbstract(){
    
    long int i;
 #ifdef DEBUG
    cout<<"forming cube"<<endl;
 #endif
    DdNode *temp_cube;
    
    cubeForAbstract=currentStateVar[STATE_ENCODE_SIZE-1];
    Cudd_Ref(cubeForAbstract);
    
    for(i=STATE_ENCODE_SIZE-2;i>=0;i--){
        temp_cube=Cudd_bddAnd(manager, cubeForAbstract, currentStateVar[i]);
        Cudd_Ref(temp_cube);
        Cudd_RecursiveDeref(manager, cubeForAbstract);
        cubeForAbstract=temp_cube;
        
    }
    for(i=INPUT_ENCODE_SIZE-1;i>=0;i--){
        temp_cube=Cudd_bddAnd(manager, cubeForAbstract, inputVar[i]);
        Cudd_Ref(temp_cube);
        Cudd_RecursiveDeref(manager, cubeForAbstract);
        cubeForAbstract=temp_cube;
    }
 #ifdef  DEBUG
    printBDD(cubeForAbstract,STATE_ENCODE_SIZE+INPUT_ENCODE_SIZE);
 #endif
}




/**
 * simulate the NFA operation
 **/

void NFAOperationByBDD::simulate(const unsigned char *buf, unsigned int len, bool check_detect){
 
    DdNode *frontier, *inputBDD;
    DdNode *nextSetOfStates;
    bool f_val;
    
    /* set current set to start set of states*/
    currentSetOfStates=startSetOfStates;
    Cudd_Ref(currentSetOfStates);

    unsigned int i;
    for(i=0;i<len;i++){
        
        inputBDD=inputSymbolBddArray[buf[i]];
        /*frontier contains current set of states and input symbol */       
        frontier=Cudd_bddAnd(manager, currentSetOfStates, inputBDD);     
        Cudd_Ref(frontier);
        
        //Cudd_RecursiveDeref(manager, frontier);
        nextSetOfStates=Cudd_bddAndAbstract(manager, transFunc ,frontier, cubeForAbstract);
        Cudd_Ref(nextSetOfStates);
        Cudd_RecursiveDeref(manager, frontier);
        
        /*make the next states to current states*/
        Cudd_RecursiveDeref(manager, currentSetOfStates);
        currentSetOfStates=Cudd_bddVarMap(manager, nextSetOfStates);
        Cudd_Ref(currentSetOfStates);
	Cudd_RecursiveDeref(manager, nextSetOfStates);	
        
        if(check_detect)
           checkAccept(i);    
    }
}



/*
 * check acceptance of the input stream
 */
void NFAOperationByBDD::checkAccept(unsigned int input_offset){
    
    int i;
    
    DdNode *result1, *result2;
    result1=Cudd_bddAnd(manager, currentSetOfStates, acceptSetOfStates);
    Cudd_Ref(result1);
    
    if(Cudd_IsNonConstant(result1)==1){
        cout<<"Accepted at offset "<< input_offset<<endl;
        
         for(i=0;i<NO_OF_PAIRS;i++){
            result2=Cudd_bddAnd(manager,currentSetOfStates, acceptStateBddArray[i]);
            
            Cudd_Ref(result2);
            if(Cudd_IsNonConstant(result2)==1) {
               cout<<" with signature "<<signatureIDArray[i]<<endl;
      	       Cudd_RecursiveDeref(manager, result2);
  	       break;
	    }
            
      	    Cudd_RecursiveDeref(manager, result2);
        }
        detection_count++;
    }     
    
    Cudd_RecursiveDeref(manager, result1);	
}


/**
 *   DEBUG PURPOSE ..REMOVE WHEN CODE IS FINALIZED
 *  prints the bdd and n is the number of variables
 *
 **/
void NFAOperationByBDD::printBDD(DdNode *bdd, int n){
    
    
    cout<<"printing bdd"<<endl;
    Cudd_PrintMinterm(manager, bdd);
   // Cudd_PrintDebug(manager, bdd, n, 4);
    
}

void NFAOperationByBDD::freeMemory(){
    if(currentStateVar!=NULL)
        free(currentStateVar);
    if(inputVar!=NULL)
        free(inputVar);
    if(nextStateVar!=NULL)
        free(nextStateVar);
    
    Cudd_Quit(manager);
    
   
}
  

void getFileName(unsigned int chunk_num, char *fileName){
     
   
   int len = strlen( bdd_src_dir); 
   strcpy(fileName, bdd_src_dir);
   strcpy(fileName + len,"/trans/transChunkBDD");   
   
   int i=3;
   char f[5];
   for(;i>=0;i--){
    f[i]=chunk_num%10+'0';
    chunk_num/=10;
   }
   f[4]='\0';
   strcat(fileName,f);
   strcat(fileName,".txt");
   
  
}

int cputime()
{
  struct rusage rus;

  getrusage (RUSAGE_SELF, &rus);
  return rus.ru_utime.tv_sec * 1000 + rus.ru_utime.tv_usec / 1000;
}



 
