/*---------------------------------------------------------------------------
 * File:	direct_simulate.cc
 *
 * Author:	Liu Yang
 * Date:	Jul 26, 2009
 *
 * Match test for different NFA implementations
 * Switch to measure the performance by CPU cycles
 *------------------------------------------------------------------------
 * $Log: direct_simulate.cc,v $
 * Revision 1.9  2010/09/27 14:21:06  lyangru
 * final synchronization
 *
 * Revision 1.1  2009/09/02 01:57:00  lyangru
 * startup
 *
 *
 *-----------------------------------------------------------------------*/


#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>

#include "nfa-multimap.h"
#include "clock.h"
#include "packet.h"
#include "decode.h"

#define BUF_SIZE		200000


//#define DEBUG
//#define DUMP_PAYLOAD_LEN

/* Choose ONE implementation when performing simulation */

//#define NFA_IMPLEMENTED_BY_SET	/* consumes most memory */
//#define NFA_IMPLEMENTED_BY_PTR_SET 	/* the fastest method */
//#define NFA_IMPLEMENTED_BY_MULTIMAP	/* consumes least memory */
#define NFA_IMPLEMENTED_BY_MAP

using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;


string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);

int main(int argc, char** argv) {

  unsigned long long starttm = 0;
  unsigned long long stoptm = 0;
  unsigned long long total_cycles = 0;
  size_t total_bytes = 0;
  unsigned int t_start, t_end, num_filled = 0;
  size_t trans_tab_sz = 0;  
#ifdef DUMP_PAYLOAD_LEN
//  ofstream out_file;
//  out_file.open("payload_len.txt");
#endif

#ifdef NFA_IMPLEMENTED_BY_MULTIMAP
  nfa_trans_mm_t nfa_dt;
#elif defined(NFA_IMPLEMENTED_BY_MAP)
  nfa_map_t nfa_dt;
#else
  nfa_trans_set_t nfa_dt; /* NFA is implemented by set or ptrs to set */
#endif

  if (argc != 3) {
    cerr << "Usage: %s <NFA file name> <trace file>" << endl;
    cout << "The NFA file should contains transitions of an NFA with epsilon transitions eliminated!" << endl;
    exit(1);
  }

  /* fill out the transition table */
#ifdef NFA_IMPLEMENTED_BY_PTR_SET
  nfa_dt.fill_ptrans_tab(argv[1]);
#else
  nfa_dt.fill_trans_tab(argv[1]);
#endif

  /* ----read stream for match test---- */
  trace_file = argv[2];

  init_pcap(&pcap_handle, trace_file);

  t_start = cputime();

  raw_packet_t rp;
  unsigned char *iphdr, *tcphdr, *payload, *prot;
  size_t tcphdr_len = 20;	/* default length of tcp header */
  size_t iphdr_len = 20;
  int payload_sz = 0;
  u_int16_t sp, dp;
  size_t total_payload_bytes = 0;
  while (get_next_packet(pcap_handle, &rp)) {
    /* locate the payload */
    iphdr = rp.pkbuf + 14;	/* skip the link hdr */
    prot = iphdr + 9;
    iphdr_len = (*iphdr & 0x0f) * 4;	/* extract the length of ip hdr */
    if (*prot == 6) {
      tcphdr = iphdr + iphdr_len;
      sp = ntohs(*(u_int16_t*) tcphdr);
      dp = ntohs(*(u_int16_t*) (tcphdr+2));
      tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
      payload = tcphdr + tcphdr_len;
      payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;

      if ((sp == 80) || (dp == 80)) { 
        if (payload_sz > 0) {
          #ifdef DUMP_PAYLOAD_LEN
  	  //out_file << payload_sz << endl;
  	  cout << payload_sz << endl;
          #endif
      #ifdef DEBUG
  	printf("payload: %s\n", payload);
      #endif
          rdtsc(starttm);
          #ifdef NFA_IMPLEMENTED_BY_PTR_SET
          nfa_dt.simulatep(payload, payload_sz);
          #else
          nfa_dt.simulate(payload, payload_sz);
          #endif
          rdtsc(stoptm);
          total_payload_bytes += payload_sz;
          total_cycles += (stoptm - starttm);
          if (rp.allocated) {
            free(rp.pkbuf);
            rp.allocated = 0;
          }
        }
      }
    } // end prot == 6
    total_bytes += rp.hdr.caplen;
  }

#ifdef DUMP_PAYLOAD_LEN
//out_file.close();
#endif

  t_end = cputime();

  cout << "Total payload bytes: " << total_payload_bytes << endl;
  cout << "Total bytes including packet headers: " << total_bytes << endl;
  cout << "Total cycles: " << total_cycles << endl;

  cout << "Simulation: execution time is " << (double)total_cycles/(double)total_payload_bytes << " cycles/byte" << endl;

  cout << "Total time spent on simulation: " << (double)(t_end - t_start)/(double)1000 << " seconds" << endl;

  return 0;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

