/*------------------------------------------------------------------------
 * File:	kgram_alpha_rdt_simul_pkt.cc
 *
 * Author:	Liu Yang
 * Date:	Sep 11, 2009
 *
 * Perform match test using real trace file based on reduced digram alphabet
 *------------------------------------------------------------------------
 * $Log: kgram_alpha_rdt_simul_pkt.cc,v $
 * Revision 1.2  2010/09/27 14:21:06  lyangru
 * final synchronization
 *
 *
 * */

#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <stdio.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <string.h>

#include "clock.h"
#include "kgram.h"
#include "packet.h"
#include "decode.h"

#define BUF_SIZE		1024
#define PAYLOAD_BUF_SIZE	2048
//#define DEBUG

using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;


string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);

int cputime();

int main(int argc, char** argv) {
  FILE *str_file = NULL;
  unsigned long long starttm = 0, stoptm = 0;
  unsigned long long total_cycles = 0;
  unsigned int t_start = 0, t_end = 0;
  char *p;
  int payload_sz = 0;
  size_t total_bytes = 0, total_payload_bytes = 0;
  char buf[BUF_SIZE];
  unsigned char len_pad = 0, act_pad = 0;
  unsigned int i, cnt = 0;
  unsigned int mapped_buf[BUF_SIZE];
  map<string, unsigned int>::iterator itm;

  kgram_nfa k_nfa;

  if (argc != 3) {
    cerr << "Usage: kgram_simul <reduced digram NFA file name> <stream file>" << endl;
    cout << "The NFA file should contains transitions of an NFA with epsilon transitions eliminated!" << endl;
    exit(1);
  }

  k_nfa.kgram_size = 2;	/* don't forget this */
  k_nfa.fill_rdt_trans_tab(argv[1]);

  /* ----read trace for match test---- */
  trace_file = argv[2];
  init_pcap(&pcap_handle, trace_file);

  raw_packet_t rp;
  unsigned char *iphdr, *tcphdr, *payload;
  unsigned char payload_buf[PAYLOAD_BUF_SIZE];
  size_t tcphdr_len = 20;	/* default length of tcp header */
  size_t iphdr_len = 20;

  while (get_next_packet(pcap_handle, &rp)) {
    /* locate the payload */
    iphdr = rp.pkbuf + 14;	/* skip the link hdr */
    iphdr_len = (*iphdr & 0x0f) * 4;	/* extract the length of ip hdr */
    tcphdr = iphdr + iphdr_len;
    tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
    payload = tcphdr + tcphdr_len;
    payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;

    if (payload_sz > 0) {
      memset(payload_buf, 0, PAYLOAD_BUF_SIZE);
      memcpy(payload_buf, payload, payload_sz);
      // decide the length of padding 
      len_pad = 2 - (unsigned char)(payload_sz % 2);
      if ((len_pad > 0) && (len_pad < 2)) {
        act_pad = len_pad;
        for (i = 0; i < len_pad; i++) {
          payload_buf[payload_sz+i] = 0;
        }
        payload_buf[payload_sz+i] = '\0';
      } else { 
        payload_buf[payload_sz] = '\0';
        act_pad = 0;
      }
      /* convert to new aphabets before simulation */
      cnt = 0;
      memset(mapped_buf, 0, BUF_SIZE * sizeof(unsigned int));
      for (i = 0; i < payload_sz; i = i + k_nfa.kgram_size) {
        string cur_dg((const char*)(payload_buf + i), (size_t)k_nfa.kgram_size);
        mapped_buf[cnt] = k_nfa.class_c[cur_dg];
        cnt++;
      }
      rdtsc(starttm);
      k_nfa.rdt_simulate2(mapped_buf, cnt);
      rdtsc(stoptm);
      total_payload_bytes += payload_sz;
      total_cycles += (stoptm - starttm);

      if (rp.allocated) {
        free(rp.pkbuf);
        rp.allocated = 0;
      }
    }
    total_bytes += rp.hdr.caplen;
  }
  
  cout << "Simulation: execution time is " << (double)total_cycles/(double)total_payload_bytes << " cycles/byte" << endl;


  return 0;
}

int cputime()
{
  struct rusage rus;

  getrusage (RUSAGE_SELF, &rus);
  return rus.ru_utime.tv_sec * 1000 + rus.ru_utime.tv_usec / 1000;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

