/*------------------------------------------------------------------------
 * File:	kgram_simul.cc
 *
 * Author:	Liu Yang
 * Date:	Sep 6, 2009
 *
 * Test multi-byte match simulation use trace file
 *------------------------------------------------------------------------
 * $Log: kgram_simul_pkt.cc,v $
 * Revision 1.3  2010/09/27 14:21:07  lyangru
 * final synchronization
 *
 *
 * */

#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <stdio.h>
#include <string.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>

#include "clock.h"
#include "kgram.h"
#include "packet.h"
#include "decode.h"

#define PAYLOAD_BUF_SIZE		2048

//#define DEBUG
using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;


string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);

int main(int argc, char** argv) {
  unsigned long long starttm = 0, stoptm = 0;
  unsigned long long total_cycles = 0;
  int payload_sz = 0;
  size_t total_bytes = 0, total_payload_bytes = 0;
  unsigned char len_pad = 0, act_pad = 0;
  unsigned i;

  kgram_nfa k_nfa;

  if (argc != 4) {
    cerr << "Usage: kgram_simul_pkt <NFA file name> <gram size> <trace file>" << endl;
    cout << "The NFA file should contains transitions of an NFA with epsilon transitions eliminated!" << endl;
    exit(1);
  }

  k_nfa.kgram_size = atoi(argv[2]);
  k_nfa.fill_trans_tab(argv[1]);
  k_nfa.construct_multi_byte_trans(k_nfa.kgram_size);
  //k_nfa.output_kgram("digram_trans");

  /* ----read stream for match test---- */
  trace_file = argv[3];
  init_pcap(&pcap_handle, trace_file);

  raw_packet_t rp;
  unsigned char *iphdr, *tcphdr, *payload;
  unsigned char payload_buf[PAYLOAD_BUF_SIZE];
  size_t tcphdr_len = 20;	/* default length of tcp header */
  size_t iphdr_len = 20;

   while (get_next_packet(pcap_handle, &rp)) {
    /* locate the payload */
    iphdr = rp.pkbuf + 14;	/* skip the link hdr */
    iphdr_len = (*iphdr & 0x0f) * 4;	/* extract the length of ip hdr */
    tcphdr = iphdr + iphdr_len;
    tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
    payload = tcphdr + tcphdr_len;
    payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;

    if (payload_sz > 0) {
      memset(payload_buf, 0, PAYLOAD_BUF_SIZE);
      memcpy(payload_buf, payload, payload_sz);
      /* decide the length of padding */
      act_pad = 0;
      len_pad = k_nfa.kgram_size - (unsigned char)(payload_sz % k_nfa.kgram_size);
      if ((len_pad > 0) && (len_pad < k_nfa.kgram_size)) {
        act_pad = len_pad;
        for (i = 0; i < len_pad; i++) {
          payload_buf[payload_sz+i] = 0;
        }
        payload_buf[payload_sz+i] = '\0';
      } else 
        payload_buf[payload_sz] = '\0';
      rdtsc(starttm);
      k_nfa.simulate2((const unsigned char *)payload_buf, payload_sz + act_pad, 2);
      rdtsc(stoptm);
      total_payload_bytes += payload_sz;
      total_cycles += (stoptm - starttm);

      if (rp.allocated) {
        free(rp.pkbuf);
        rp.allocated = 0;
      }
    }
    total_bytes += rp.hdr.caplen;
  }
  
  cout << "Simulation: execution time is " << (double)total_cycles/(double)total_payload_bytes << " cycles/byte" << endl;


  return 0;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

