/*---------------------------------------------------------------------------
 * File:	trace_trans.cc
 *
 * Author:	Liu Yang
 * Date:	June 11, 2010
 *
 * Extract the nature of transactions, e.g., number of users, web requests, etc., performed in a trace file
 *------------------------------------------------------------------------
 * $Log: trace_trans.cc,v $
 * Revision 1.1  2010/09/27 14:22:54  lyangru
 * extract traffic features
 *
 *
 *
 *-----------------------------------------------------------------------*/


#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>

#include "nfa-multimap.h"
#include "clock.h"
#include "packet.h"
#include "decode.h"

#define BUF_SIZE		200000


//#define DEBUG
//#define DUMP_PAYLOAD_LEN

using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;

string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);
char *addr_to_str(uint32_t x);

int main(int argc, char** argv) {

  unsigned long long starttm = 0;
  unsigned long long stoptm = 0;
  unsigned long long total_cycles = 0;
  size_t total_bytes = 0;
  unsigned int t_start, t_end, num_filled = 0;
  size_t trans_tab_sz = 0;  
  ofstream out_file2;
#ifdef DUMP_PAYLOAD_LEN
  ofstream out_file;
  out_file.open("payload_len.txt");
#endif

  if (argc != 3) {
    cerr << "Usage: trace_trans <trace file> <output filename for IPs>" << endl;
    exit(1);
  }

  out_file2.open(argv[2]);
  /* ----read stream for match test---- */
  trace_file = argv[1];
  init_pcap(&pcap_handle, trace_file);

  raw_packet_t rp;
  unsigned char *iphdr, *tcphdr, *payload;
  size_t tcphdr_len = 20;	/* default length of tcp header */
  size_t iphdr_len = 20;
  int payload_sz = 0;
  size_t total_payload_bytes = 0;
  uint32_t src_addr, dst_addr;

  while (get_next_packet(pcap_handle, &rp)) {
    /* locate the payload */
    iphdr = rp.pkbuf + 14;	/* skip the link hdr */
    iphdr_len = (*iphdr & 0x0f) * 4;	/* extract the length of ip hdr */
    tcphdr = iphdr + iphdr_len;
    tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
    payload = tcphdr + tcphdr_len;
    payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;

    src_addr = *(uint32_t *)(iphdr + 12);    
    dst_addr = *(uint32_t *)(iphdr + 16);    
    //printf("ip header src: %s, ", addr_to_str(src_addr));
    //printf("dst: %s\n", addr_to_str(dst_addr));
    out_file2 << addr_to_str(src_addr) << endl;
    out_file2 << addr_to_str(dst_addr) << endl;

    if (payload_sz > 0) {
#ifdef DEBUG
  printf("payload: %s\n", payload);
#endif
#ifdef DUMP_PAYLOAD_LEN
  out_file << payload_sz << endl;
#endif
      total_payload_bytes += payload_sz;
      if (rp.allocated) {
        free(rp.pkbuf);
        rp.allocated = 0;
      }
    }
    total_bytes += rp.hdr.caplen;
  }

#ifdef DUMP_PAYLOAD_LEN
out_file.close();
#endif
  out_file2.close();

  return 0;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

char *addr_to_str(uint32_t x) {
#if 1
  static char str[16];
  (void) memset(str, 0, 16);
#else
  char *str = malloc(16);
#endif
  sprintf(str, "%u.%u.%u.%u",
	 ((unsigned char *)&x)[0],
	 ((unsigned char *)&x)[1],
	 ((unsigned char *)&x)[2],
	 ((unsigned char *)&x)[3]);
  return str;
}

