/*-----------------------------------------------------------------------------
 * File:    nfa_h
 *
 * TODO:
 *   (1) add code to do epsilon elimination only
 *   (2) cleanup 'lookup' function in minimize.cc
 *   (3) fold minimize.cc into nfa_t as a class member.
 *
 * Author:  Randy Smith
 * Date:    18 May 2007
 *
 *
 *    Copyright 2006,2007 Randy Smith, smithr@cs.wisc.edu
 *
 *    This file contains unpublished confidential proprietary
 *    work of Randy Smith, Department of Computer Sciences,
 *    University of Wisconsin--Madison.  No use of any sort, including
 *    execution, modification, copying, storage, distribution, or reverse
 *    engineering is permitted without the express written consent of
 *    Randy Smith.
 *
 *-----------------------------------------------------------------------------
 * History
 * $Log: nfa.h,v $
 * Revision 1.5  2010/09/27 14:48:31  lyangru
 * final synchronization
 *
 * Revision 1.1  2009/06/09 18:51:22  vinodg
 * *** empty log message ***
 *
 * Revision 1.4  2008/04/17 23:08:34  smithr
 * Added code to convert from dfa_tab_t to nfa_t
 *
 * Revision 1.3  2008/04/17 21:12:26  smithr
 * Added routines to construct an efficient table representing/matching
 * DFAs.
 *
 * Revision 1.2  2008/02/13 20:33:10  smithr
 * *** empty log message ***
 *
 * Revision 1.1  2007/08/07 18:00:53  smithr
 * initial check in to CVS
 *
 *
 *---------------------------------------------------------------------------*/
#ifndef NFA_H
#define NFA_H

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include <vector>
#include <list>
#include <bitset>
#include <set>

typedef unsigned int state_id_t;

class nfa_state_t 
{
   public:
      nfa_state_t(void);
      nfa_state_t(const nfa_state_t& copy);
      ~nfa_state_t(void);

      nfa_state_t& operator=(const nfa_state_t& rhs);

      std::list<state_id_t> trans[256];
      std::list<state_id_t> eps_trans;

      /* In combined, minimized machines, I may have
       * multiple accept id's at a single state 
       * TODO: consider making these unsigned */
      std::list<int> accept_id;    

      void print_state(FILE *out, unsigned int indent);

};


class nfa_t 
{
   public:
      nfa_t(void);
      ~nfa_t(void);
      
      std::vector<nfa_state_t> states;
      unsigned int start;
      std::list<state_id_t> accepting;
      state_id_t next_id;

      unsigned int machine_id;

      /* deep clones the existing NFA */
      nfa_t* clone(void);

      state_id_t add_state(void);
      int   add_trans(state_id_t from, state_id_t to, unsigned char sym);
      int   add_eps_trans(state_id_t from, state_id_t to);
      int   add_eps_trans(const std::list<state_id_t>& from, state_id_t to);
      int   add_eps_trans(state_id_t from, const std::list<state_id_t>& to);
      unsigned int fold_in(nfa_t *nfa,state_id_t& ns,std::list<state_id_t>&na);
      /* same as the fold_in() function, but keeping the accept_id's 
       * for the purpose of signature matching
       * added by Liu Yang */
      unsigned int fold_in_acc(nfa_t *nfa, state_id_t&ns, 
			    std::list<state_id_t>& na);
      void set_accepting_states(int id);

      void print_machine(FILE *out);
      void xfa_output(const char *filename, unsigned int sig_id=0);
      void xfa_output(FILE *out, unsigned int sig_id=0);

      /* output an nfa/dfa to be used by BDD project 
       * added by Liu Yang */
      void bdd_output(const char *filename, unsigned int sid_id=0);
      void bdd_output(FILE *out, unsigned int sig_id=0);


      nfa_t* make_dfa(void);
      bool is_deterministic(void) const;
      
      void set_id(unsigned int id);
      void set_accepting_states(unsigned int id);

  private:
      int compute_eps_closure(const std::set<state_id_t>& T, 
			      std::set<state_id_t>& out);
      int compute_eps_closure(state_id_t s, std::set<state_id_t>& out);
      int compute_move(const std::set<state_id_t>& T, unsigned int a,
		       std::set<state_id_t>& out);

      void output_char_hack(FILE *out, unsigned int c);
};


typedef struct dfa_tab_t
{
      bool populate(const nfa_t& nfa);
      bool populate_dfa_from_file(char* fileName, unsigned int dfa_count);	
      nfa_t *to_nfa_t(void);


      void cleanup(void);
      void simulate(const unsigned char *buf, unsigned int len, 
		    bool intermed) const;

      unsigned int machine_id;
      unsigned int num_states;
      unsigned int start;
      unsigned int **tab;

      /* since we are recording accept ids, we need to store more than
       * just a boolean "yes" or "no" */
      int ** acc;
} dfa_tab_t;



/* inductive step routines (these combine existing nfas appropriately) */
nfa_t* do_kleene_closure(nfa_t *input);
nfa_t *do_concat(nfa_t *left, nfa_t *right);
nfa_t *do_alternation(nfa_t *left, nfa_t *right);
nfa_t *do_positive_closure(nfa_t *input);
nfa_t *do_repetition_count_single(nfa_t *input, unsigned int count);
nfa_t *do_repetition_count_range(nfa_t *in, unsigned int lo, unsigned int hi);
nfa_t *do_repetition_count_unrestricted_max(nfa_t *in, unsigned int lo);
nfa_t *do_repetition_count_unrestricted_min(nfa_t *in, unsigned int max);
nfa_t *do_at_most_once(nfa_t *in);

/*-----------------------------------------------------------------------------
 * do_alternation_acc -- implements the OR (|) operator 
 * and keeps the accepting states of the source NFA's
 * The goal to do this is to keep track of the accepting states and accept_id's
 * which will be used in signature matching
 * added by Liu Yang
 *---------------------------------------------------------------------------*/
nfa_t *do_alternation_acc(nfa_t *left, nfa_t *right);

/* basis step routines (these build nfas from scratch) */
nfa_t *make_nfa_char(const std::bitset<256>& chars);
nfa_t *make_nfa_eps(void);
nfa_t *make_nfa_empty(void);
nfa_t *make_nfa_caret(void);

#endif
