
/**------------------------------------------------------------------------
 * File:	nfaToBDD.cpp
 *
 * Author:	Rezwana Karim
 * Date:	August 20, 2009
 *----------------------------------------------------------------------
 * 1. Read epsilon-free NFA information from file 
 * 2. Encode the NFA using binary encoding
 * 3. Construct BDDs corresponding to 
 *        transition function
 *        set of start states
 *        set of accepting states
 *        individual accept states
 *        input symbols
 * 4. Dumps the BDDs to disk       
 *  
 *  @input
 *       fileName and location where the BDDs will be dumped
 *  
 *        
 *
 *------------------------------------------------------------------------
 *
 *
 **/

#include <iostream>
#include <fstream>
#include <cstdlib>
#include <time.h>
#include <cmath>
#include <bitset>
#include <vector>
#include <map>
#include <set>
#include <sys/time.h>
#include <sys/resource.h>

using namespace std;

/*Set these file paths appropriately based on where the CUDD package lies*/

#include "../../cudd-2.4.2/include/cuddObj.hh"
#include "../../cudd-2.4.2/include/cuddInt.h"
#include "../../cudd-2.4.2/dddmp/dddmpInt.h"

#include <boost/dynamic_bitset.hpp>

#include "nfaToBDD.h"

#define DEBUG
#define TRACE
#define MAX_TRANSITIONS 1000000

char dest_dir[300];


int cputime();

/**
 * Read the metadata from the epsilon free NFA transition table
 **/

FILE* NFAtoBDD::readNFAFromFile(char *fileName ){
    
    FILE *fp;
    fp=fopen(fileName,"r");
    if(!fp){
       cout<<"cannot open file "<<fileName<<endl;
       exit(1);
    }   
  #ifdef DEBUG
    cout<<"successfully opened file"<<endl;
  #endif    
    
    fscanf(fp,"%u",&NO_OF_STATES);
  #ifdef DEBUG
    cout<<" Number of states : "<<NO_OF_STATES<<endl;
  #endif
     
    STATE_ENCODE_SIZE=(unsigned short)ceil(log10(NO_OF_STATES)/log10(2));
  #ifdef DEBUG
    cout<<" STATE_ENCODE_SIZE="<<STATE_ENCODE_SIZE<<endl;
  #endif
    
    unsigned int i, j;
    
    fscanf(fp,"%u",&NO_OF_PAIRS);
  #ifdef DEBUG
    cout<<" No of Pairs  "<<NO_OF_PAIRS<<endl;
  #endif
    
    ssiPair=new stateSigIDPair_t[NO_OF_PAIRS];
    
    for(i=0;i<NO_OF_PAIRS;i++){
        fscanf(fp,"%u",&j);
        ssiPair[i].state=j;
        fscanf(fp,"%u",&j);
        ssiPair[i].sigID=j;
      #ifdef DEBUG
        cout<<"pair "<<i<<" "<<ssiPair[i].state<<" "<<ssiPair[i].sigID<<endl;
      #endif
    }

    fscanf(fp,"%u",&NO_OF_ACC_STATES);
  #ifdef DEBUG
    cout<<" No of ACCepting STATES  "<<NO_OF_ACC_STATES<<endl;
  #endif
    acceptSetOfStates= new unsigned int[NO_OF_ACC_STATES];
    
    for(i = 0; i< NO_OF_ACC_STATES; i++){

        fscanf(fp,"%u",&acceptSetOfStates[i]);
      #ifdef DEBUG
        cout<<" Accepting state   "<< i << " value is "<<acceptSetOfStates[i]<<endl;
      #endif
    }

    fscanf(fp,"%u",&NO_OF_START_STATES);
  #ifdef DEBUG
    cout<<" No of STARTING STATES  "<<NO_OF_START_STATES<<endl;
  #endif
    startSetOfStates= new unsigned int[NO_OF_START_STATES];
    
    for(i = 0; i< NO_OF_START_STATES; i++){
         fscanf(fp,"%u",&startSetOfStates[i]);       
      #ifdef DEBUG
        cout<<" Starting state   "<< i << " value is "<<startSetOfStates[i]<<endl;
      #endif
    }
    
    fscanf(fp,"%u",&NO_OF_TRANSITIONS);       
  #ifdef  DEBUG
    cout<<" No:of TRANSITIONS IN FILE  "<<NO_OF_TRANSITIONS<<endl;
  #endif
 
    return fp;
}

/**
 * Initialize the BDD construction framework
 * Create neessary BDD variables and assign an ordering
 **/

void NFAtoBDD::initBddConstruct(){
   
    
    int numberOfVars=2*STATE_ENCODE_SIZE+INPUT_ENCODE_SIZE;
     
     /*----------------------- Init Var Information Structure ------------------*/

    varInfo.nDdVars = numberOfVars;

    varInfo.rootNames = NULL;
    varInfo.ddType = DDDMP_NONE;
    varInfo.nVars = (-1);
    varInfo.nSuppVars = (-1);
    varInfo.varNamesFlagUpdate = 1;
    varInfo.suppVarNames = NULL;
    varInfo.orderedVarNames = NULL;
    varInfo.varIdsFlagUpdate = 1;
    varInfo.varIds = NULL;
    varInfo.varIdsAll = NULL;
    varInfo.varComposeIdsFlagUpdate = 1;
    varInfo.varComposeIds = NULL;
    varInfo.varComposeIdsAll = NULL;
    varInfo.varAuxIdsFlagUpdate = 1;
    varInfo.varAuxIds = NULL;
    varInfo.varAuxIdsAll = NULL;
    varInfo.nRoots = (-1);

    /** Initialize the manager
      * numberOfVars     : Initial number of BDD variables
      * 0                : Initial number of ZDD variables
      * CUDD_UNIQUE_SLOTS: Initial size of the unique tables (256)
      * CUDD_CACHE_SLOTS : Initial size of the cache (262144)
      * 0                : target maximum memory (0 means unlimited)
      **/
   
    manager = Cudd_Init(numberOfVars, 0, CUDD_UNIQUE_SLOTS, CUDD_CACHE_SLOTS, 0);
     
    /*allocating the bdd variables*/
    currentStateVar= new DdNode* [STATE_ENCODE_SIZE];
    nextStateVar=new DdNode* [STATE_ENCODE_SIZE];
    inputVar=new DdNode*[INPUT_ENCODE_SIZE];
    
    /*setting the ordering of the BDD variables. Here the order is i<x<y */
    unsigned short int i;
    
    for(i=0;i<STATE_ENCODE_SIZE;i++){
        
        currentStateVar[i]=Cudd_bddIthVar(manager, INPUT_ENCODE_SIZE + STATE_ENCODE_SIZE-1-i);
        nextStateVar[i]=Cudd_bddIthVar(manager, STATE_ENCODE_SIZE+INPUT_ENCODE_SIZE+STATE_ENCODE_SIZE-1-i);
    }
    
    for(i=0;i<INPUT_ENCODE_SIZE;i++)
        inputVar[i]=Cudd_bddIthVar(manager,  INPUT_ENCODE_SIZE-1-i);
    
    return;
}


/** Read MAX_TRANSITIONS from the file
 *    convert them to binary
 *    form BDD
 *    dump that a file
 * 
 **/


int  NFAtoBDD::processTransitions(FILE *fp){

    unsigned int ps,input,ns;
    unsigned long int t;
    binCurrentState = new boost::dynamic_bitset<>* [MAX_TRANSITIONS];
    if(!binCurrentState){
                cout<<"allocaiton error of binCurrentState"<<endl;
                exit(1);
    }
    binNextState = new boost::dynamic_bitset<>* [MAX_TRANSITIONS];
    if(!binNextState){
                cout<<"allocaiton error of binNextState"<<endl;
                exit(1);
    }
    binInput = new boost::dynamic_bitset<>* [MAX_TRANSITIONS];
    if(!binInput){
                cout<<"allocaiton error of binInput "<<endl;
                exit(1);
    }
    
    unsigned int i;
    for(i=0;i<MAX_TRANSITIONS;i++) {
            binCurrentState[i]=new boost::dynamic_bitset<> (STATE_ENCODE_SIZE);
            if(!binCurrentState[i]){
                cout<<"allocaiton error of binCurrentState[i] "<<i<<endl;
                exit(1);
            }
            binNextState[i]=new boost::dynamic_bitset<> (STATE_ENCODE_SIZE);
            if(!binNextState[i]){
                cout<<"allocaiton error of binNextState[i] "<<i<<endl;
                exit(1);
            }
            binInput[i]=new boost::dynamic_bitset<> (INPUT_ENCODE_SIZE);
            if(!binInput[i]){
                cout<<"allocaiton error of binInput[i] "<<i<<endl;
                exit(1);
            }
    }
    
    short int chunk_number=0;
    t=0;
    while (1){
        for(i=0;t<NO_OF_TRANSITIONS&&i<MAX_TRANSITIONS;i++,t++) {
            /*read transitions form file*/
            fscanf(fp,"%u %u %u",&ps,&input,&ns);
        
            /*create binary encoding */
            boost::dynamic_bitset<>temp_cs(STATE_ENCODE_SIZE, ps);
            boost::dynamic_bitset<>temp_input(INPUT_ENCODE_SIZE, input);
            boost::dynamic_bitset<>temp_ns(STATE_ENCODE_SIZE, ns);
            
            /*sstore in the binary table*/
            *binCurrentState[i]=temp_cs;
            *binNextState[i]=temp_ns;
            *binInput[i]=temp_input;
                      
          #ifdef TRACE
            if(t%10000==0)
               cout<<t<<" th transitions processed for Binary encoding at time "<<cputime()<<endl;
          #endif
            
        }  
        chunk_number++;
        /*create nfa bdd for chunk of 1 million transitions*/
        DdNode *transFuncChunk=createTransitionChunkBDD(i);    
        
        /*dump that to file*/
        storeTransFuncChunkBDD( transFuncChunk, chunk_number);
        
         if(t==NO_OF_TRANSITIONS)
                break;        
    }   
    
    fclose(fp);  
    
    /*free memory*/
    
    for(i=0;i<MAX_TRANSITIONS;i++){
    
        delete binCurrentState[i];
        delete binInput[i];
        delete binNextState[i];
    }    
        
    
    if(binCurrentState!=NULL)
        free(binCurrentState);
    if(binNextState!=NULL)
        free(binNextState);
    if(binInput!=NULL)
        free(binInput);
    
    return 0;

}

/**
 * creates transition function for the NFA
 *@return
 *      transition function expressed as a BDD
 */


DdNode* NFAtoBDD::createTransitionChunkBDD(unsigned int CHUNK_SIZE){
    
  
    unsigned int i;
    unsigned short int j;
    DdNode *minterm, *tempMinterm, *tempOR, *transFunction;
    
    /*intitalize to constant 0*/
    transFunction=Cudd_Not( Cudd_ReadOne(manager));
    Cudd_Ref(transFunction);
    
    /* Construct the transiiton function*/
    for(i=0;i<CHUNK_SIZE;i++){
        
        if((*binCurrentState[i])[0]==1){
            minterm=currentStateVar[0];
        }
        else
            minterm=Cudd_Not(currentStateVar[0]);
        Cudd_Ref(minterm);
        
        for(j=1;j<STATE_ENCODE_SIZE;j++){
            if((*binCurrentState[i])[j]==1)
                tempMinterm=Cudd_bddAnd(manager, minterm, currentStateVar[j]);
            else
                tempMinterm=Cudd_bddAnd(manager, minterm, Cudd_Not(currentStateVar[j]));
            
            Cudd_Ref(tempMinterm);
            Cudd_RecursiveDeref(manager, minterm);
            minterm=tempMinterm;
        }
        
        for(j=0;j<INPUT_ENCODE_SIZE;j++){
            if((*binInput[i])[j]==1)
                tempMinterm=Cudd_bddAnd(manager, minterm, inputVar[j]);
            else
                tempMinterm=Cudd_bddAnd(manager, minterm, Cudd_Not(inputVar[j]));
                
            Cudd_Ref(tempMinterm);
            Cudd_RecursiveDeref(manager, minterm);
            minterm=tempMinterm;
         }
            
         for(j=0;j<STATE_ENCODE_SIZE;j++){
             if((*binNextState[i])[j]==1)
                 tempMinterm=Cudd_bddAnd(manager, minterm, nextStateVar[j]);
                    
             else
                 tempMinterm=Cudd_bddAnd(manager, minterm, Cudd_Not(nextStateVar[j]));
                    
             Cudd_Ref(tempMinterm);
             Cudd_RecursiveDeref(manager, minterm);
             minterm=tempMinterm;
         }
                
         tempOR=Cudd_bddOr(manager, transFunction, minterm);
         Cudd_Ref(tempOR);
         Cudd_RecursiveDeref(manager, transFunction);
         transFunction=tempOR;
         Cudd_RecursiveDeref(manager, minterm);        
        
      #ifdef TRACE
         if(i%10000==0)
           cout<<i<<" th Transition processed for BDD CONSTRUCTION at time "<<cputime()<<endl;
      #endif
      }
     
      return transFunction;
}
        
/**
 * form BDD representing set of Starting States
 **/
   
DdNode* NFAtoBDD::formStartSetOfStates(){
           
            
    unsigned long int i, j;
    DdNode *minterm, *temp_minterm, *temp_or, *startSetOfStatesNode;
    
 #ifdef TRACE
    int s_count=0;
 #endif
    
    
    startSetOfStatesNode =  Cudd_ReadLogicZero(manager);
    
       
    for(i=0;i<NO_OF_START_STATES;i++){
         /*Getting the binary encoding of a single start set*/
         boost::dynamic_bitset<>temp_ns(STATE_ENCODE_SIZE,  (unsigned long) startSetOfStates[i]);
         /*constructing bdd*/
         if(temp_ns[0])
            minterm=currentStateVar[0];
         else
            minterm=Cudd_Not(currentStateVar[0]);
                    
         Cudd_Ref(minterm);
                    
         for(j=1;j<STATE_ENCODE_SIZE;j++){
            if(temp_ns[j])
                temp_minterm=Cudd_bddAnd(manager, minterm, currentStateVar[j]);
            else
                temp_minterm=Cudd_bddAnd(manager, minterm, Cudd_Not(currentStateVar[j]));
                        
            Cudd_Ref(temp_minterm);
            Cudd_RecursiveDeref(manager, minterm);
            minterm=temp_minterm;
         }
         temp_or=Cudd_bddOr(manager, startSetOfStatesNode, minterm );
         Cudd_Ref(temp_or);
         Cudd_RecursiveDeref(manager, minterm);
         Cudd_RecursiveDeref(manager, startSetOfStatesNode);
         startSetOfStatesNode=temp_or;
         
      #ifdef TRACE
         s_count++;  
         if(s_count%100==0)
             cout<<s_count<<" th state processed for START SET OF STATES at time "<<cputime()<<endl;
      #endif
    }
            
    return startSetOfStatesNode;
}
        
/**
  * form BDD representing set of Accepting States
  **/
DdNode* NFAtoBDD::formAcceptSetOfStates(){
            
    unsigned long int i, j;
    DdNode *minterm, *temp_minterm, *temp_or, *acceptSetOfStatesNode;
    acceptStateBddArray=new DdNode*[NO_OF_ACC_STATES];        
    
  #ifdef TRACE
    int a_count=0;
  #endif
   
            
    acceptSetOfStatesNode=Cudd_ReadLogicZero(manager);       
    for(i=0;i<NO_OF_ACC_STATES;i++){
        /*Getting the binary encoding of a single accept set*/
        boost::dynamic_bitset<>temp_ns(STATE_ENCODE_SIZE, (unsigned long)acceptSetOfStates[i]);
        /*constructing bdd*/
        if(temp_ns[0])
              minterm=currentStateVar[0];
         else
              minterm=Cudd_Not(currentStateVar[0]);
         Cudd_Ref(minterm);
                
          for(j=1;j<STATE_ENCODE_SIZE;j++){
                 if(temp_ns[j])
                     temp_minterm=Cudd_bddAnd(manager, minterm, currentStateVar[j]);
                 else
                     temp_minterm=Cudd_bddAnd(manager, minterm, Cudd_Not(currentStateVar[j]));
                    
                 Cudd_Ref(temp_minterm);
                 Cudd_RecursiveDeref(manager, minterm);
                 minterm=temp_minterm;
           }
           acceptStateBddArray[i]=minterm;
           /*debug : Cudd_PrintMinterm(manager,acceptStateBddArray[i]);*/
           temp_or=Cudd_bddOr(manager, acceptSetOfStatesNode, minterm );
           Cudd_Ref(temp_or);
           Cudd_RecursiveDeref(manager, minterm);
           Cudd_RecursiveDeref(manager, acceptSetOfStatesNode);
           acceptSetOfStatesNode=temp_or;
           
        #ifdef TRACE
           a_count++;  
           if(a_count%100==0)
              cout<<a_count<<" th state processed for ACCEPT SET OF STATES at time "<<cputime()<<endl;
        #endif
                
     }
            
     return acceptSetOfStatesNode;
}
        

DdNode** NFAtoBDD::formAcceptStateBddArray(){
     
    DdNode **acceptStateBddArray;// array of bdds representing each accepting state ...needed for detection of signature id after a match
    unsigned long int i, j;
    DdNode *minterm, *temp_minterm;
    acceptStateBddArray=new DdNode*[NO_OF_PAIRS];              
            
    #ifdef TRACE
       int p_count=0;
    #endif
       
    for(i=0;i<NO_OF_PAIRS;i++){
        /*Getting the binary encoding of a single accept set*/
        boost::dynamic_bitset<>temp_ns(STATE_ENCODE_SIZE, (unsigned long)ssiPair[i].state);
        
        /*construct individual bdd*/
        if(temp_ns[0])
              minterm=currentStateVar[0];
         else
              minterm=Cudd_Not(currentStateVar[0]);
         Cudd_Ref(minterm);
                
          for(j=1;j<STATE_ENCODE_SIZE;j++){
                 if(temp_ns[j])
                     temp_minterm=Cudd_bddAnd(manager, minterm, currentStateVar[j]);
                 else
                     temp_minterm=Cudd_bddAnd(manager, minterm, Cudd_Not(currentStateVar[j]));
                    
                 Cudd_Ref(temp_minterm);
                 Cudd_RecursiveDeref(manager, minterm);
                 minterm=temp_minterm;
           }
           acceptStateBddArray[i]=minterm;
           
            
          #ifdef TRACE
              p_count++;  
              if(p_count%100==0)
                cout<<p_count<<" th PAIR processed at time "<<cputime()<<endl;
           #endif
     }
            
     return acceptStateBddArray;
    
}
        
        
void NFAtoBDD::printTable(){
    
  unsigned int i;
 
  
  for (i = 0; i < NO_OF_TRANSITIONS; i++) {
    cout << "state[" << T_transitions[i][0] << "]" <<endl;
    cout << "symbol = " << T_transitions[i][1] << "-->" << " state[" << T_transitions[i][2] << "]" << endl;
  }  
     
}  
  

void NFAtoBDD::printBinTable(){
    
    unsigned int i,j;
    
    for(i=0;i<NO_OF_TRANSITIONS;i++){
         
         for(j=STATE_ENCODE_SIZE-1;j>=0;j--){
             cout<<(*binCurrentState[i])[j]<<" ";
         }
         cout<<" ";
         for(j=INPUT_ENCODE_SIZE-1;j>=0;j--){
             cout<<(*binInput[i])[j]<<" ";
         }
         cout<<" ";
         for(j=STATE_ENCODE_SIZE-1;j>=0;j--){
             cout<<(*binNextState[i])[j]<<" ";
         }
         
         cout<<endl;
     }
    
}

/**stores 
 *  encoding size of a state
 *  encoding size of an input symbol
 *  number of <accept state, signature id > pair
 *  all the <accept state, signature id > pairs
 * in txt file
 */
void NFAtoBDD::storeNFAInfo(){
    
    char infoFile[500];
    
    int len = strlen(dest_dir); 
    strcpy(infoFile, dest_dir);
    strcpy(infoFile + len,"/metadata/nfa_info.txt");


    ofstream outputFile;
    outputFile.open(infoFile);
    outputFile<< STATE_ENCODE_SIZE<<endl<<INPUT_ENCODE_SIZE;
    
    outputFile<<endl<<NO_OF_PAIRS<<endl;
    unsigned int i;
    for(i=0;i<NO_OF_PAIRS;i++){
        
        outputFile<<ssiPair[i].sigID<<endl;
    }       
    outputFile.close();
    
}

/**
 * Stores i.e. dump the BDDs representing NFA transition function into disk
 **/

int NFAtoBDD::storeTransFuncChunkBDD(DdNode* transFunc, int chunk_num){
   /*get fileName*/
   char *fileName=new char[500];
   getFileName(chunk_num, fileName);
   
   int ret_value= Dddmp_cuddBddStore (manager, NULL, transFunc,
        varInfo.orderedVarNames, varInfo.varAuxIdsAll, DDDMP_MODE_TEXT,
        varoutinfo, fileName, NULL);
   
    Cudd_RecursiveDeref(manager, transFunc);
    free(fileName);
    return ret_value;
}

/**
 * Stores i.e. dump the BDDs representing set of start states of the NFA into disk
 **/

void NFAtoBDD::storeStartSetOfStatesBDD(DdNode* start){
 
    char fileName[500];
    
    int len = strlen(dest_dir); 
    strcpy(fileName, dest_dir);
    strcpy(fileName + len,"/metadata/start.txt");
    

    Dddmp_cuddBddStore (manager, NULL, start,
        varInfo.orderedVarNames, varInfo.varAuxIdsAll, DDDMP_MODE_TEXT,
        varoutinfo, fileName, NULL);
 
}

/**
 * Stores i.e. dump the BDDs representing set of accept states of the NFA into disk
 **/

void NFAtoBDD::storeAcceptSetOfStatesBDD(DdNode* accept){
    
    char fileName[500];
    
    int len = strlen(dest_dir); 
    strcpy(fileName, dest_dir);
    strcpy(fileName + len,"/metadata/accept.txt");



    Dddmp_cuddBddStore (manager, NULL, accept,
        varInfo.orderedVarNames, varInfo.varAuxIdsAll, DDDMP_MODE_TEXT,
        varoutinfo, fileName, NULL);
    
}

/** 
 * Stores array of bdds representing accepting  states 
 **/
void NFAtoBDD::storeAcceptStateBddArray(DdNode **acceptStateBddArray){
    
    char fileName[500];
    
    int len = strlen(dest_dir); 
    strcpy(fileName, dest_dir);
    strcpy(fileName + len,"/metadata/accept_bdd_array.txt");
 
    Dddmp_cuddBddArrayStore (manager, NULL, NO_OF_PAIRS, acceptStateBddArray,
    NULL, varInfo.orderedVarNames, varInfo.varAuxIdsAll, DDDMP_MODE_TEXT,
    DDDMP_VARIDS, fileName, NULL);
}
   
/** 
 * Stores array of bdds representing input symbols from the alphabet 
 **/

void NFAtoBDD::storeInputAlphabetBddArray(){
    
    
    DdNode **inputSymbolBddArray= new DdNode* [INPUT_ALPHABET_SIZE];
    
    unsigned short int symbol;
    unsigned char i;
    DdNode *minterm, *temp_minterm;
    
    for(symbol=0;symbol<INPUT_ALPHABET_SIZE;symbol++){
         boost::dynamic_bitset<>temp_input(INPUT_ENCODE_SIZE, (unsigned long)symbol);
         
    
         if(temp_input[0])
              minterm=inputVar[0];
         else
              minterm=Cudd_Not(inputVar[0]);
         Cudd_Ref(minterm);
    
         for(i=1;i<INPUT_ENCODE_SIZE;i++){
               if(temp_input[i])
                    temp_minterm=Cudd_bddAnd(manager, minterm, inputVar[i]);
                else
                    temp_minterm=Cudd_bddAnd(manager, minterm, Cudd_Not(inputVar[i]));
        
               Cudd_Ref(temp_minterm);
               Cudd_RecursiveDeref(manager, minterm);
               minterm=temp_minterm;
         }
         inputSymbolBddArray[symbol]=minterm;
    
    }
   
    char fileName[500];    
    int len = strlen(dest_dir); 
    strcpy(fileName, dest_dir);
    strcpy(fileName + len,"/metadata/input_bdd_array.txt");
 
    Dddmp_cuddBddArrayStore (manager, NULL, INPUT_ALPHABET_SIZE, inputSymbolBddArray,
        NULL, varInfo.orderedVarNames, varInfo.varAuxIdsAll, DDDMP_MODE_TEXT,
        DDDMP_VARIDS, fileName, NULL);
    
    if(!inputSymbolBddArray)
        free(inputSymbolBddArray);

}

/**
 * Free allocated memory
 **/ 

void NFAtoBDD::freeMemory( DdNode **acceptStateBddArray){

        
    if(currentStateVar!=NULL)
        free(currentStateVar);
    if(inputVar!=NULL)
        free(inputVar);
    if(nextStateVar!=NULL)
        free(nextStateVar);
    if(acceptSetOfStates!=NULL)
        free(acceptSetOfStates);
   
    
    if(acceptStateBddArray!=NULL)
        free(acceptStateBddArray);
      
    Cudd_Quit(manager);

}

/**
 * computes the cputime 
 **/
int cputime()
{
  struct rusage rus;

  getrusage (RUSAGE_SELF, &rus);
  return rus.ru_utime.tv_sec * 1000 + rus.ru_utime.tv_usec / 1000;
}

/**
 * Get the filename where the BDD corresponding to i-th chunk of the tranisiton function will be stored
 **/
void getFileName(unsigned int chunk_num, char *fileName){
   
   int len = strlen(dest_dir); 
   strcpy(fileName, dest_dir);
   strcpy(fileName + len,"/trans/transChunkBDD");
   
   int i=3;
   char f[5];
   for(;i>=0;i--){
    f[i]=chunk_num%10+'0';
    chunk_num/=10;
   }
   f[4]='\0';
   strcat(fileName,f);
   strcat(fileName,".txt");
   
  
}



int main(int argc, char *argv[]){
            
    NFAtoBDD ntbObj;
            
   
    if(argc<=2){
        cout<< "USAGE ::" << argv[0] <<" <NFA transition file name> <path to destination directory for generated BDDs>"<< endl;
        return 1;
    }

 #ifdef TRACE
    cout<<"NFA Header reading started from FILE"<<endl;
 #endif
    
    FILE *fp=ntbObj.readNFAFromFile(argv[1]);
    strcpy(dest_dir, argv[2]);

    
    rootmatchmode = DDDMP_ROOT_MATCHLIST;
 #if 1
    varmatchmode = DDDMP_VAR_MATCHIDS;
 #else
    varmatchmode = DDDMP_VAR_MATCHNAMES;
 #endif
    varoutinfo = DDDMP_VARIDS;
    
 #ifdef TRACE
    cout<<"Finished READING header info"<<endl;
    cout<<"intitializing BDD CONSTRUCTION "<<endl;
 #endif
    
    ntbObj.initBddConstruct();  
    /*start processing chunk of transitions */
    int start_time=cputime();
     
 #ifdef TRACE
    cout<<"Finished READING header info"<<endl;
    cout<<"intitializing BDD CONSTRUCTION "<<endl;
 #endif
 
    ntbObj.processTransitions(fp); 

   /*track starting itme of bdd creation for the nfa*/
    
 #ifdef PRINT_TRANS_FUNC
    cout<<"transiiton function"<<endl;
    Cudd_PrintDebug(ntbObj.manager, transFunc,2*ntbObj.STATE_ENCODE_SIZE+ntbObj.INPUT_ENCODE_SIZE,4);
 #endif
   
    DdNode *start=ntbObj.formStartSetOfStates();
 #ifdef PRINT_START_STATE
    cout<<"start state"<<endl;
    Cudd_PrintDebug(ntbObj.manager, start,ntbObj.STATE_ENCODE_SIZE,4);
 #endif
    
 #ifdef TRACE
    cout<<"Finished BDD CONSTRUCTION of set of start states"<<endl;
    cout<<"starting storing start state"<<endl;
 #endif
    
    ntbObj.storeStartSetOfStatesBDD(start);
    
 #ifdef TRACE
    cout<<"finished storing start state"<<endl;
    cout<<"starting BDD CONSTRUCTION of set of ACCEPT states"<<endl;
 #endif
    
    DdNode *accept=ntbObj.formAcceptSetOfStates();
 #ifdef ACCEPT_STATE
    cout<<"accept state"<<endl;
    Cudd_PrintDebug(ntbObj.manager, accept,ntbObj.STATE_ENCODE_SIZE,4);
 #endif
    
 #ifdef TRACE
    cout<<"Finished BDD CONSTRUCTION of set of ACCEPT states"<<endl;
    cout<<"starting storing ACCEPT BDD"<<endl;  
 #endif
    
    ntbObj.storeAcceptSetOfStatesBDD(accept);
    
 #ifdef TRACE
    cout<<"finished storing ACCEPT BDD"<<endl;  
    cout<<"starting BDD CONSTRUCTION of set of PAIRS"<<endl;
 #endif
    
    DdNode **acceptStateBddArray=ntbObj.formAcceptStateBddArray();
    
    int end_time=cputime();
    
    ntbObj.storeNFAInfo();
    
 #ifdef TRACE
    cout<<"Finished BDD Storage "<<endl;
    cout<<"starting Storing BDD for pairs "<<endl;
 #endif
    ntbObj.storeAcceptStateBddArray(acceptStateBddArray); 
    
 #ifdef TRACE
    cout<<"Finished BDD Storage of pairs "<<endl;
    cout<<"starting Storing BDD for input symbols "<<endl;
 #endif
    
    ntbObj.storeInputAlphabetBddArray();
    
    cout << "BDD construction time is " << ((double)(end_time - start_time)/1000)<<" s" << endl; 
    ntbObj.freeMemory( acceptStateBddArray); 
   
    return 0;        
}



 
