/* ----------------------------------------------------------------------------
 * packet.h
 *   Definitions for a decoded packet.  Packets are received in pcap format
 *   and decoded into the packet structure defined here.
 *
 *
 * ----------------------------------------------------------------------------
 * History:
 *  $Log: packet.h,v $
 *  Revision 1.1  2009/08/31 21:18:07  lyangru
 *  startup
 *
 *  Revision 1.1.1.1  2006/02/24 22:10:58  smithr
 *  ips research code
 *
 *  Revision 1.6  2005/08/17 02:58:26  smithr
 *  Changes necessary to get the ids running on the sun machines.  The two
 *  biggest set of changes involved:
 *  (1) packing all the data structures to get the alignment issues resolved
 *  (2) getting the definitions of u_int16_t, etc., defined correctly.
 *
 *  Revision 1.5  2005/08/15 13:12:16  smithr
 *  Basic detection is working.
 *
 *  Revision 1.4  2005/07/31 02:43:35  smithr
 *  all plugins checked in, code compiles.
 *
 *  Revision 1.3  2005/07/21 16:53:37  smithr
 *  Added IP protocol number struct so that we can see the protocol number.
 *
 *  Revision 1.2  2005/07/20 23:24:50  smithr
 *  Miscellaneous updates; working on verifying flow functionality
 *
 *  Revision 1.1  2005/07/19 16:30:16  smithr
 *  Initial check-in to CVS.  packet/decoding framework for ethernet
 *  packets is working.
 *
 *
 * --------------------------------------------------------------------------*/
#ifndef PACKET_H
#define PACKET_H

#include <sys/types.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>

#include "types.h"
/* struct to collect packet statistics */
typedef struct _PacketCount
{
    u_long total;

    u_long other;
    u_long tcp;
    u_long udp;
    u_long icmp;
    u_long arp;
    u_long eapol;
    u_long ipv6;
    u_long ipx;
    u_long discards;
    u_long alert_pkts;
    u_long log_pkts;
    u_long pass_pkts;

    u_long frags;           /* number of frags that have come in */
    u_long frag_trackers;   /* number of tracking structures generated */
    u_long rebuilt_frags;   /* number of packets rebuilt */
    u_long frag_incomp;     /* number of frags cleared due to memory issues */
    u_long frag_timeout;    /* number of frags cleared due to timeout */
    u_long rebuild_element; /* frags that were element of rebuilt pkt */
    u_long frag_mem_faults; /* number of times the memory cap was hit */

    u_long tcp_stream_pkts; /* number of packets tcp reassembly touches */
    u_long rebuilt_tcp;     /* number of phoney tcp packets generated */
    u_long tcp_streams;     /* number of tcp streams created */
    u_long rebuilt_segs;    /* number of tcp segments used in rebuilt pkts */
    u_long str_mem_faults;  /* number of times the stream memory cap was hit */

  /* wireless statistics */
    u_long wifi_mgmt;
    u_long wifi_data;
    u_long wifi_control; 
    u_long assoc_req;
    u_long assoc_resp;
    u_long reassoc_req;
    u_long reassoc_resp;
    u_long probe_req;
    u_long probe_resp;
    u_long beacon;
    u_long atim;
    u_long dissassoc;
    u_long auth;
    u_long deauth;
    u_long ps_poll;
    u_long rts;
    u_long cts;
    u_long ack;
    u_long cf_end;
    u_long cf_end_cf_ack;
    u_long data;
    u_long data_cf_ack;
    u_long data_cf_poll;
    u_long data_cf_ack_cf_poll;
    u_long cf_ack;
    u_long cf_poll;
    u_long cf_ack_cf_poll;
} PacketCount;



/* A lot of these are taken from decode.h, pared down to deal with
 * Ethernet only */



/*  D E F I N E S  ***********************************************************/
#define ETHERNET_MTU                  1500
#define ETHERNET_TYPE_IP              0x0800
#define ETHERNET_TYPE_ARP             0x0806
#define ETHERNET_TYPE_REVARP          0x8035
#define ETHERNET_TYPE_EAPOL           0x888e
#define ETHERNET_TYPE_IPV6            0x86dd
#define ETHERNET_TYPE_IPX             0x8137
#define ETHERNET_TYPE_PPPoE_DISC      0x8863 /* discovery stage */
#define ETHERNET_TYPE_PPPoE_SESS      0x8864 /* session stage */
#define ETHERNET_TYPE_8021Q           0x8100

#define ETH_DSAP_SNA                  0x08    /* SNA */
#define ETH_SSAP_SNA                  0x00    /* SNA */
#define ETH_DSAP_STP                  0x42    /* Spanning Tree Protocol */
#define ETH_SSAP_STP                  0x42    /* Spanning Tree Protocol */
#define ETH_DSAP_IP                   0xaa    /* IP */
#define ETH_SSAP_IP                   0xaa    /* IP */

#define ETH_ORG_CODE_ETHR              0x000000    /* Encapsulated Ethernet */
#define ETH_ORG_CODE_CDP               0x00000c    /* Cisco Discovery Proto */

#define ETHERNET_HEADER_LEN             14
#define ETHERNET_MAX_LEN_ENCAP          1518    /* 802.3 (+LLC) or ether II ? */
#define PPPOE_HEADER_LEN                20
#define MINIMAL_TOKENRING_HEADER_LEN    22
#define MINIMAL_IEEE80211_HEADER_LEN    10    /* Ack frames and others */
#define IEEE802_11_DATA_HDR_LEN         24    /* Header for data packets */
#define TR_HLEN                         MINIMAL_TOKENRING_HEADER_LEN
#define TOKENRING_LLC_LEN                8
#define SLIP_HEADER_LEN                 16


/* otherwise defined in /usr/include/ppp_defs.h */
#define IP_HEADER_LEN           20
#define TCP_HEADER_LEN          20
#define UDP_HEADER_LEN          8
#define ICMP_HEADER_LEN         4


/* ip option type codes */
#ifndef IPOPT_EOL
    #define IPOPT_EOL            0x00
#endif

#ifndef IPOPT_NOP
    #define IPOPT_NOP            0x01
#endif

#ifndef IPOPT_RR
    #define IPOPT_RR             0x07
#endif

#ifndef IPOPT_RTRALT
    #define IPOPT_RTRALT         0x14
#endif

#ifndef IPOPT_TS
    #define IPOPT_TS             0x44
#endif

#ifndef IPOPT_SECURITY
    #define IPOPT_SECURITY       0x82
#endif

#ifndef IPOPT_LSRR
    #define IPOPT_LSRR           0x83
#endif

#ifndef IPOPT_LSRR_E
    #define IPOPT_LSRR_E         0x84
#endif

#ifndef IPOPT_SATID
    #define IPOPT_SATID          0x88
#endif

#ifndef IPOPT_SSRR
    #define IPOPT_SSRR           0x89
#endif






#define TCP_OPTLENMAX           40 /* (((2^4) - 1) * 4  - TCP_HEADER_LEN) */

#ifndef IP_MAXPACKET
#define	IP_MAXPACKET	65535		/* maximum packet size */
#endif /* IP_MAXPACKET */

#define TH_FIN  0x01
#define TH_SYN  0x02
#define TH_RST  0x04
#define TH_PUSH 0x08
#define TH_ACK  0x10
#define TH_URG  0x20
#define TH_RES2 0x40
#define TH_RES1 0x80
#define TH_NORESERVED (TH_FIN|TH_SYN|TH_RST|TH_PUSH|TH_ACK|TH_URG)

/* http://www.iana.org/assignments/tcp-parameters
 *
 * tcp options stuff. used to be in <netinet/tcp.h> but it breaks
 * things on AIX
 */
#define TCPOPT_EOL              0   /* End of Option List [RFC793] */
#define TCPOLEN_EOL             1   /* Always one byte */

#define TCPOPT_NOP              1   /* No-Option [RFC793] */
#define TCPOLEN_NOP             1   /* Always one byte */

#define TCPOPT_MAXSEG           2   /* Maximum Segment Size [RFC793] */
#define TCPOLEN_MAXSEG          4   /* Always 4 bytes */

#define TCPOPT_WSCALE           3   /* Window scaling option [RFC1323] */
#define TCPOLEN_WSCALE          3   /* 1 byte with logarithmic values */

#define TCPOPT_SACKOK           4   /* Experimental [RFC2018]*/
#define TCPOLEN_SACKOK          2

#define TCPOPT_SACK             5   /* Experimental [RFC2018] variable length*/

#define TCPOPT_ECHO             6   /* Echo (obsoleted by option 8) [RFC1072]*/
#define TCPOLEN_ECHO            6   /* 6 bytes  */

#define TCPOPT_ECHOREPLY        7   /* Echo Reply (obs. by option 8)[RFC1072]*/
#define TCPOLEN_ECHOREPLY       6   /* 6 bytes  */

#define TCPOPT_TIMESTAMP        8   /* Timestamp [RFC1323], 10 bytes */
#define TCPOLEN_TIMESTAMP       10

#define TCPOPT_PARTIAL_PERM     9   /* Partial Order Permitted/
				                   Experimental [RFC1693] */
#define TCPOLEN_PARTIAL_PERM    2   /* Partial Order Permitted/ 
				                   Experimental [RFC1693] */

#define TCPOPT_PARTIAL_SVC      10  /*  Partial Order Profile [RFC1693] */
#define TCPOLEN_PARTIAL_SVC     3   /*  3 bytes long -- Experimental */

/* atleast decode T/TCP options... */
#define TCPOPT_CC               11  /*  T/TCP Connection count  [RFC1644] */
#define TCPOPT_CC_NEW           12  /*  CC.NEW [RFC1644] */
#define TCPOPT_CC_ECHO          13  /*  CC.ECHO [RFC1644] */
#define TCPOLEN_CC             6  /* page 17 of rfc1644 */
#define TCPOLEN_CC_NEW         6  /* page 17 of rfc1644 */
#define TCPOLEN_CC_ECHO        6  /* page 17 of rfc1644 */

#define TCPOPT_ALTCSUM          15  /* TCP Alternate Checksum Data 
				                 [RFC1146], variable length */
#define TCPOPT_SKEETER          16  /* Skeeter [Knowles] */
#define TCPOPT_BUBBA            17  /* Bubba   [Knowles] */

#define TCPOPT_TRAILER_CSUM     18  /*Trailer Checksum Option [Subbu&Monroe]*/
#define TCPOLEN_TRAILER_CSUM  3  

#define TCPOPT_MD5SIG           19  /* MD5 Signature Option [RFC2385] */
#define TCPOLEN_MD5SIG        18

/* Space Communications Protocol Standardization */
#define TCPOPT_SCPS             20  /* Capabilities [Scott] */
#define TCPOPT_SELNEGACK        21  /* Selective Negative Ack [Scott] */
#define TCPOPT_RECORDBOUND         22  /* Record Boundaries [Scott] */
#define TCPOPT_CORRUPTION          23  /* Corruption experienced [Scott] */

#define TCPOPT_SNAP                24  /* SNAP [Sukonnik]--anyone have info?*/
#define TCPOPT_UNASSIGNED          25  /* Unassigned (released 12/18/00) */
#define TCPOPT_COMPRESSION         26  /* TCP Compression Filter [Bellovin] */
/* http://www.research.att.com/~smb/papers/draft-bellovin-tcpcomp-00.txt*/


#define TCP_OPT_TRUNC -1
#define TCP_OPT_BADLEN -2


/*
 * Default maximum segment size for TCP.
 * With an IP MSS of 576, this is 536,
 * but 512 is probably more convenient.
 * This should be defined as MIN(512, IP_MSS - sizeof (struct tcpiphdr)).
 */

#ifndef TCP_MSS
    #define    TCP_MSS      512
#endif

#ifndef TCP_MAXWIN
    #define    TCP_MAXWIN   65535    /* largest value for (unscaled) window */
#endif

#ifndef TCP_MAX_WINSHIFT 
    #define TCP_MAX_WINSHIFT    14    /* maximum window shift */
#endif

/*
 * User-settable options (used with setsockopt).
 */
#ifndef TCP_NODELAY
    #define    TCP_NODELAY   0x01    /* don't delay send to coalesce packets */
#endif

#ifndef TCP_MAXSEG
    #define    TCP_MAXSEG    0x02    /* set maximum segment size */
#endif

#define SOL_TCP        6    /* TCP level */

#if defined(__GNUC__)
/* force word-aligned ntohl parameter */
    #define EXTRACT_32BITS(p)  ({ u_int32_t __tmp; memmove(&__tmp, (p), sizeof(u_int32_t)); (u_int32_t) ntohl(__tmp);})

#else

/* allows unaligned ntohl parameter - dies w/SIGBUS on SPARCs */
    #define EXTRACT_32BITS(p) ((u_int32_t) ntohl (*(u_int32_t *)(p)))

#endif                /* WORDS_MUSTALIGN */



#define PKT_REBUILT_FRAG     0x00000001  /* is a rebuilt fragment */
#define PKT_REBUILT_STREAM   0x00000002  /* is a rebuilt stream */
#define PKT_STREAM_UNEST_UNI 0x00000004  /* is from an unestablished stream and
                                          * we've only seen traffic in one
                                          * direction
                                          */
#define PKT_STREAM_UNEST_BI  0x00000008  /* is from an unestablished stream and
                                          * we've seen traffic in both 
                                          * directions
                                          */
#define PKT_STREAM_EST       0x00000010  /* is from an established stream */
#define PKT_ECN              0x00000020  /* this is ECN traffic */
#define PKT_FROM_SERVER      0x00000040  /* this packet came from the server
                                            side of a connection (TCP) */
#define PKT_FROM_CLIENT      0x00000080  /* this packet came from the client
                                            side of a connection (TCP) */
#define PKT_HTTP_DECODE      0x00000100  /* this packet has normalized http */
#define PKT_FRAG_ALERTED     0x00000200  /* this packet has been alerted by 
                                            defrag */
#define PKT_STREAM_INSERT    0x00000400  /* this packet has been inserted 
					                        into stream4 */
#define PKT_ALT_DECODE       0x00000800  /* this packet has been normalized by
					    by (only set when we must look at 
					    an alernative buffer) */
#define PKT_STREAM_TWH       0x00001000
#define PKT_INLINE_DROP      0x20000000
#define PKT_OBFUSCATED       0x40000000  /* this packet has been obfuscated */
#define PKT_LOGGED           0x80000000  /* this packet has been logged */

typedef struct _Options
{
    u_int8_t code;
    u_int8_t len; /* length of the data section */
    u_int8_t *data;
} __attribute__ ((packed)) Options;


/* TCP Flag definitions */
#define R_FIN          0x01
#define R_SYN          0x02
#define R_RST          0x04
#define R_PSH          0x08
#define R_ACK          0x10
#define R_URG          0x20
#define R_RES2         0x40
#define R_RES1         0x80


/* ICMP Definitions */
#define ICMP_ECHOREPLY          0    /* Echo Reply                   */
#define ICMP_DEST_UNREACH       3    /* Destination Unreachable      */
#define ICMP_SOURCE_QUENCH      4    /* Source Quench                */
#define ICMP_REDIRECT           5    /* Redirect (change route)      */
#define ICMP_ECHO               8    /* Echo Request                 */
#define ICMP_ROUTER_ADVERTISE   9    /* Router Advertisement         */
#define ICMP_ROUTER_SOLICIT     10    /* Router Solicitation          */
#define ICMP_TIME_EXCEEDED      11    /* Time Exceeded                */
#define ICMP_PARAMETERPROB      12    /* Parameter Problem            */
#define ICMP_TIMESTAMP          13    /* Timestamp Request            */
#define ICMP_TIMESTAMPREPLY     14    /* Timestamp Reply              */
#define ICMP_INFO_REQUEST       15    /* Information Request          */
#define ICMP_INFO_REPLY         16    /* Information Reply            */
#define ICMP_ADDRESS            17    /* Address Mask Request         */
#define ICMP_ADDRESSREPLY       18    /* Address Mask Reply           */
#define NR_ICMP_TYPES           18

/* Codes for ICMP UNREACHABLES */
#define ICMP_NET_UNREACH        0    /* Network Unreachable          */
#define ICMP_HOST_UNREACH       1    /* Host Unreachable             */
#define ICMP_PROT_UNREACH       2    /* Protocol Unreachable         */
#define ICMP_PORT_UNREACH       3    /* Port Unreachable             */
#define ICMP_FRAG_NEEDED        4    /* Fragmentation Needed/DF set  */
#define ICMP_SR_FAILED          5    /* Source Route failed          */
#define ICMP_NET_UNKNOWN        6
#define ICMP_HOST_UNKNOWN       7
#define ICMP_HOST_ISOLATED      8
#define ICMP_PKT_FILTERED_NET   9
#define ICMP_PKT_FILTERED_HOST  10
#define ICMP_NET_UNR_TOS        11
#define ICMP_HOST_UNR_TOS       12
#define ICMP_PKT_FILTERED       13    /* Packet filtered */
#define ICMP_PREC_VIOLATION     14    /* Precedence violation */
#define ICMP_PREC_CUTOFF        15    /* Precedence cut off */
#define NR_ICMP_UNREACH         15    /* instead of hardcoding immediate
                                       * value */

#define ICMP_REDIR_NET          0
#define ICMP_REDIR_HOST         1
#define ICMP_REDIR_TOS_NET      2
#define ICMP_REDIR_TOS_HOST     3

#define ICMP_TIMEOUT_TRANSIT    0
#define ICMP_TIMEOUT_REASSY     1

#define ICMP_PARAM_BADIPHDR     0
#define ICMP_PARAM_OPTMISSING   1
#define ICMP_PARAM_BAD_LENGTH   2




/* ---------------------------------------------------------------------------
 * Now, finally, here are the header definitions 
 * 
 * -------------------------------------------------------------------------*/

typedef struct protos_t
{
      unsigned char num;
      const char *str;
      const char *descr;
} protos_t;

extern protos_t proto_map[];

/* Do I need these vlan headers? not sure... */
#define VTH_PRIORITY(vh)  ((ntohs((vh)->vth_pri_cfi_vlan) & 0xe000) >> 13)
#define VTH_CFI(vh)	  ((ntohs((vh)->vth_pri_cfi_vlan & 0x0100) >> 12))
#define VTH_VLAN(vh)	  ((ntohs((vh)->vth_pri_cfi_vlan) & 0x0FFF))

typedef struct _VlanTagHdr
{
    u_int16_t vth_pri_cfi_vlan;
    u_int16_t vth_proto;  /* protocol field... */
} VlanTagHdr;


/* 
 * Ethernet header
 */

typedef struct _EtherHdr
{
    u_int8_t ether_dst[6];
    u_int8_t ether_src[6];
    u_int16_t ether_type;

} __attribute__ ((packed)) EtherHdr;


/* tcpdump shows us the way to cross platform compatibility */
#define IP_VER(iph)	(((iph)->ip_verhl & 0xf0) >> 4)
#define IP_HLEN(iph)	((iph)->ip_verhl & 0x0f)

/* we need to change them as well as get them */
#define SET_IP_VER(iph, value)  ((iph)->ip_verhl = (((iph)->ip_verhl & 0x0f) | (value << 4)))
#define SET_IP_HLEN(iph, value)  ((iph)->ip_verhl = (((iph)->ip_verhl & 0xf0) | (value & 0x0f)))

typedef struct _IPHdr
{
    u_int8_t ip_verhl;      /* version & header length */
    u_int8_t ip_tos;        /* type of service */
    u_int16_t ip_len;       /* datagram length */
    u_int16_t ip_id;        /* identification  */
    u_int16_t ip_off;       /* fragment offset */
    u_int8_t ip_ttl;        /* time to live field */
    u_int8_t ip_proto;      /* datagram protocol */
    u_int16_t ip_csum;      /* checksum */
    struct in_addr ip_src;  /* source IP */
    struct in_addr ip_dst;  /* dest IP */
} __attribute__ ((packed)) IPHdr;


/* more macros for TCP offset */
#define TCP_OFFSET(tcph)	(((tcph)->th_offx2 & 0xf0) >> 4)
#define TCP_X2(tcph)	((tcph)->th_offx2 & 0x0f)

/* we need to change them as well as get them */
#define SET_TCP_OFFSET(tcph, value)  ((tcph)->th_offx2 = (((tcph)->th_offx2 & 0x0f) | (value << 4)))
#define SET_TCP_X2(tcph, value)  ((tcph)->th_offx2 = (((tcph)->th_offx2 & 0xf0) | (value & 0x0f)))

typedef struct _TCPHdr
{
    u_int16_t th_sport;     /* source port */
    u_int16_t th_dport;     /* destination port */
    u_int32_t th_seq;       /* sequence number */
    u_int32_t th_ack;       /* acknowledgement number */
    u_int8_t th_offx2;      /* offset and reserved */
    u_int8_t th_flags;
    u_int16_t th_win;       /* window */
    u_int16_t th_sum;       /* checksum */
    u_int16_t th_urp;       /* urgent pointer */

} __attribute__ ((packed)) TCPHdr;


typedef struct _UDPHdr
{
    u_int16_t uh_sport;
    u_int16_t uh_dport;
    u_int16_t uh_len;
    u_int16_t uh_chk;

} __attribute__ ((packed)) UDPHdr;


/* ---------------------------------------------
 * ICMP Header structure definition
 *
 * -------------------------------------------*/
#define s_icmp_pptr       icmp_hun.pptr
#define s_icmp_gwaddr     icmp_hun.gwaddr
#define s_icmp_id         icmp_hun.idseq.id
#define s_icmp_seq        icmp_hun.idseq.seq
#define s_icmp_void       icmp_hun.sih_void
#define s_icmp_pmvoid     icmp_hun.pmtu.ipm_void
#define s_icmp_nextmtu    icmp_hun.pmtu.nextmtu
#define s_icmp_num_addrs  icmp_hun.rtradv.num_addrs
#define s_icmp_wpa        icmp_hun.rtradv.wpa
#define s_icmp_lifetime   icmp_hun.rtradv.lifetime

#define s_icmp_otime      icmp_dun.ts.otime
#define s_icmp_rtime      icmp_dun.ts.rtime
#define s_icmp_ttime      icmp_dun.ts.ttime
#define s_icmp_ip         icmp_dun.ih_ip
#define s_icmp_radv       icmp_dun.radv
#define s_icmp_mask       icmp_dun.mask
#define s_icmp_data       icmp_dun.data

typedef struct idseq
{
      u_int16_t id;
      u_int16_t seq;
} idseq_t;

typedef struct _ICMPHdr
{
      u_int8_t type;
      u_int8_t code;
      u_int16_t csum;
      union
      {
	    u_int8_t pptr;
	    
	    struct in_addr gwaddr;
	    
	    idseq_t idseq;
	    
	    int sih_void;
	    
	    struct pmtu 
	    {
		  u_int16_t ipm_void;
		  u_int16_t nextmtu;
	    } pmtu;
	    
	    struct rtradv 
	    {
		  u_int8_t num_addrs;
		  u_int8_t wpa;
		  u_int16_t lifetime;
	    } rtradv;
      } icmp_hun;
      

      union 
      {
	    /* timestamp */
	    struct ts 
	    {
		  u_int32_t otime;
		  u_int32_t rtime;
		  u_int32_t ttime;
	    } ts;
	    
	    /* IP header for unreach */
	    struct ih_ip  
	    {
		  IPHdr *ip;
		  /* options and then 64 bits of data */
	    } ip;
	    
	    struct ra_addr 
	    {
		  u_int32_t addr;
		  u_int32_t preference;
	    } radv;
	    
	    u_int32_t mask;
	    
	    char    data[1];
	    
      } icmp_dun;


} __attribute__ ((packed)) ICMPHdr;

//RDS -- is this an ICMP echo request structure?
typedef struct _echoext
{
    u_int16_t id;
    u_int16_t seqno;

} echoext;






//forward reference.  See flow.h for definition
struct _FLOW;

typedef struct _packet
{
      struct pcap_pkthdr pkth;   /* BPF data */
      u_int8_t *pkt;              /* base pointer to the raw packet data */
      
      EtherHdr *eh;               /* standard TCP/IP/Ethernet/ARP headers */
      VlanTagHdr *vh;
      
      IPHdr *iph, *orig_iph;   /* and orig. headers for ICMP_*_UNREACH family*/
      u_int32_t ip_options_len;
      u_int8_t *ip_options_data;
      
      TCPHdr *tcph, *orig_tcph;
      u_int32_t tcp_options_len;
      u_int8_t *tcp_options_data;
      
      UDPHdr *udph, *orig_udph;
      ICMPHdr *icmph, *orig_icmph;

      //echoext *ext;           /* ICMP echo extension struct */

      u_int8_t *data;         /* packet payload pointer */
      u_int16_t dsize;        /* packet payload size */
      u_int16_t alt_dsize;    /* the dsize of a packet before munging 
				 (used for log)*/

      u_int16_t actual_ip_len;/* for logging truncated packets (usually by 
				 a small snaplen) */
      
      u_int8_t frag_flag;     /* flag to indicate a fragmented packet */
      u_int16_t frag_offset;  /* fragment offset number */
      u_int8_t mf;            /* more fragments flag */
      u_int8_t df;            /* don't fragment flag */
      u_int8_t rf;            /* IP reserved bit */
      
      u_int16_t sp;           /* source port (TCP/UDP) */
      u_int16_t dp;           /* dest port (TCP/UDP) */
      u_int16_t orig_sp;      /* source port (TCP/UDP) of original datagram */
      u_int16_t orig_dp;      /* dest port (TCP/UDP) of original datagram */
      u_int32_t caplen;       /* capture packet len (from decode.c) */
      
      u_int8_t uri_count;     /* number of URIs in this packet */
      
      void *ssnptr;           /* for tcp session tracking info... */
      struct _FLOW *flow;     /* for flow info */
      void *streamptr;        /* for tcp pkt dump */
      
      Options ip_options[40]; /* ip options decode structure */
      u_int32_t ip_option_count;  /* number of options in this packet */
      u_char ip_lastopt_bad;  /* flag to indicate that option decoding was
				 halted due to a bad option */
      Options tcp_options[TCP_OPTLENMAX];    /* tcp options decode struct */
      u_int32_t tcp_option_count;
      u_char tcp_lastopt_bad;  /* flag to indicate that option decoding was
				  halted due to a bad option */
      
      //u_int8_t csum_flags;        /* checksum flags */
      u_int32_t packet_flags;     /* special flags for the packet */
      int preprocessors;          /* flags for preprocessors to check */

      unsigned long id;        /*unique id of packet */

} __attribute__ ((packed)) packet;


/* miscellaneous output functions */
void dump_packet(FILE *stream, const packet *p);
void dump_ip(FILE *stream, const IPHdr *iph);
void dump_tcp(FILE *stream, const TCPHdr *tcph);
void dump_udp(FILE *stream, const UDPHdr *udph);




#endif
