/*---------------------------------------------------------------------------
 * File:	direct_rdt_simul.cc
 *
 * Author:	Liu Yang
 * Date:	Sep 10, 2009
 *
 * Match test for different NFA implementations
 * using reduced aphabet.
 * The reduction only support nfa_map_t for now.
 *------------------------------------------------------------------------
 * $Log: direct_rdt_simul.cc,v $
 * Revision 1.2  2010/09/27 14:21:06  lyangru
 * final synchronization
 *
 *
 *
 *-----------------------------------------------------------------------*/


#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <vector>
#include <string>
#include <cstdlib>
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <pcap.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <errno.h>

#include "nfa-multimap.h"
#include "clock.h"
#include "packet.h"
#include "decode.h"

#define BUF_SIZE		200000

//#define DEBUG

/* Choose ONE implementation when performing simulation */

//#define NFA_IMPLEMENTED_BY_SET	/* consumes most memory */
//#define NFA_IMPLEMENTED_BY_PTR_SET 	/* the fastest method */
//#define NFA_IMPLEMENTED_BY_MULTIMAP	/* consumes least memory */
#define NFA_IMPLEMENTED_BY_MAP

using namespace std;

typedef struct raw_packet
{
      int                allocated; // >0 iff pkt points to allocated data
      unsigned char      *pkbuf;
      packet             pkt;
      unsigned long      id;
      struct pcap_pkthdr hdr;
} raw_packet_t;


string trace_file;

pcap_t *pcap_handle;

/*-----------------------------------------------
 * Function Prototypes
 *----------------------------------------------*/

void init_pcap(pcap_t **handle, string& fname);
int get_next_packet(pcap_t *handle, raw_packet_t* pkt);

int main(int argc, char** argv) {

  unsigned long long starttm = 0;
  unsigned long long stoptm = 0;
  unsigned long long total_cycles = 0;
  size_t total_bytes = 0;
  unsigned int t_start, t_end, num_filled = 0;
  size_t trans_tab_sz = 0; 
  nfa_map_t nfa_dt; 
/*
#ifdef NFA_IMPLEMENTED_BY_MULTIMAP
  nfa_trans_mm_t nfa_dt;
#elif defined(NFA_IMPLEMENTED_BY_MAP)
  nfa_map_t nfa_dt;
#else
  nfa_trans_set_t nfa_dt; 
#endif
*/
  if (argc != 3) {
    cerr << "Usage: %s <NFA file name> <trace file>" << endl;
    cout << "The NFA file should contains transitions of an NFA with epsilon transitions eliminated!" << endl;
    exit(1);
  }

  /* fill out the transition table */
/*
#ifdef NFA_IMPLEMENTED_BY_PTR_SET
  nfa_dt.fill_ptrans_tab(argv[1]);
#else
  nfa_dt.fill_trans_tab(argv[1]);
#endif */

  nfa_dt.fill_trans_tab(argv[1]);
  /* perform alphabet reduction */
  nfa_dt.alphabet_reduction_im();

  /* generate the reduced transition table */
  nfa_dt.rdt_trans_tab_gen();
 
  /* ----read stream for match test---- */
  trace_file = argv[2];

  init_pcap(&pcap_handle, trace_file);

  t_start = cputime();

  raw_packet_t rp;
  unsigned char *iphdr, *tcphdr, *payload;
  size_t tcphdr_len = 20;	/* default length of tcp header */
  size_t iphdr_len = 20;
  int payload_sz = 0;
  size_t total_payload_bytes = 0;
  while (get_next_packet(pcap_handle, &rp)) {
    /* locate the payload */
    iphdr = rp.pkbuf + 14;	/* skip the link hdr */
    iphdr_len = (*iphdr & 0x0f) * 4;	/* extract the length of ip hdr */
    tcphdr = iphdr + iphdr_len;
    tcphdr_len = ((*(tcphdr + 12) >> 4) & 0x0f) * 4;
    payload = tcphdr + tcphdr_len;
    payload_sz = rp.hdr.caplen - 14 - iphdr_len - tcphdr_len;

    if (payload_sz > 0) {
      rdtsc(starttm);
      nfa_dt.rdt_simulate(payload, payload_sz);
      rdtsc(stoptm);
      total_payload_bytes += payload_sz;
      total_cycles += (stoptm - starttm);
      if (rp.allocated) {
        free(rp.pkbuf);
        rp.allocated = 0;
      }
    }
    total_bytes += rp.hdr.caplen;
  }

  t_end = cputime();

  cout << "Total payload bytes: " << total_payload_bytes << endl;
  cout << "Total bytes including packet headers: " << total_bytes << endl;
  cout << "Total cycles: " << total_cycles << endl;

  cout << "Simulation: execution time is " << (double)total_cycles/(double)total_payload_bytes << " cycles/byte" << endl;

  cout << "Total time spent on simulation: " << (double)(t_end - t_start)/(double)1000 << " seconds" << endl;

  return 0;
}

/*--------------------------------
 * init_pcap
 * -------------------------------*/
void init_pcap(pcap_t **handle, string& fname)
{
   char errbuf[PCAP_ERRBUF_SIZE];

   *handle = pcap_open_offline(fname.c_str(), errbuf);
   if (handle == NULL)
   {
      fprintf(stderr, "pcap_open failed: %s\n", errbuf);
      exit(0);
   }
   fprintf(stdout, "Opened trace file %s\n", fname.c_str());
}

/*--------------------------------------
 * get_next_packet
 *-------------------------------------*/
int get_next_packet(pcap_t *handle, raw_packet_t* pkt)
{
   static int id=0;
   const u_int8_t *data;

   data = pcap_next(handle, &(pkt->hdr));
   if (!data)
      return 0;

   pkt->pkbuf = (u_int8_t *)calloc(1, pkt->hdr.caplen);
   memcpy(pkt->pkbuf, data, pkt->hdr.caplen);
   pkt->allocated = 1;
   return 1;
}

