/*-----------------------------------------------------------------------------
 *main.cc
 * Simple program for testing the pcre parser.
 *
 * Author:  Randy Smith
 * Date:    10 August 2006
 *
 *
 *    Copyright 2006,2007 Randy Smith, smithr@cs.wisc.edu
 *
 *    This file contains unpublished confidential proprietary
 *    work of Randy Smith, Department of Computer Sciences,
 *    University of Wisconsin--Madison.  No use of any sort, including
 *    execution, modification, copying, storage, distribution, or reverse
 *    engineering is permitted without the express written consent of
 *    Randy Smith.
 *
 *-----------------------------------------------------------------------------
 * History:
 * $Log: main.cc,v $
 * Revision 1.4  2010/09/27 14:48:31  lyangru
 * final synchronization
 *
 * Revision 1.1  2009/06/09 18:51:22  vinodg
 * *** empty log message ***
 *
 * Revision 1.2  2008/02/13 20:33:10  smithr
 * *** empty log message ***
 *
 * Revision 1.1  2007/08/07 18:00:50  smithr
 * initial check in to CVS
 *
 * Revision 1.4  2007/05/14 20:48:03  smithr
 * Added:
 * (1) disallowing greedy operators *? +? ??
 * (2) Added tree-printing code.
 *
 * Revision 1.3  2006/08/11 14:55:30  smithr
 * Added parse_utils, which provides a mechanism for parsing regexes stored
 * in a string rather than just stored in a file.
 *
 * Revision 1.2  2006/08/10 18:35:34  smithr
 * Several additions and bug fixes, including:
 * (1) user defined char classes implemented.
 * (2) escape sequences fully handled
 * (3) memory management addressed (destructors free mem as necessary)
 * (4) many other miscellaneous bugs, etc.
 *
 *
 *---------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include "nodes.h"
#include "re.h"
#include "nfa.h"
#include "minimize.h"
#include "globals.h"

void usage_and_die(const char *msg);
nfa_t *dfa_min1();

/* this is defined in combine_alg.cc */
nfa_t* combine(nfa_t* NFA1, nfa_t* NFA2);


regex_tree_t *RE = NULL;
nfa_t *NFA = NULL;
nfa_t *DFA = NULL;

/*-----------------------------------------------------------------------------
 * main
 *---------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
   FILE *f;
   char buf[4096];
   char *p;
   int line = 1;
   vector<nfa_t *> VEC_NFA, VEC_DFA;
   nfa_t *tmp_nfa, *com_nfa, *com_dfa;
   nfa_t *tmp_dfa, *com_dfa2;
   char dfa_file[][10] = {"dfa00.txt", "dfa01.txt", "dfa02.txt", "dfa03.txt", "dfa04.txt", "dfa05.txt", "dfa06.txt"};
   char nfa_file[][10] = {"nfa00.txt", "nfa01.txt", "nfa02.txt", "nfa03.txt", "nfa04.txt", "nfa05.txt", "nfa06.txt"};
   /* bitd(n)faxx.txt represent a dfa(nfa) of the example on page 65 of the automata book */   
//   char dfa_file[][15] = {"bitdfa00.txt", "bitdfa01.txt", "bitdfa02.txt", "bitdfa03.txt", "bitdfa04.txt", "bitdfa05.txt", "bitdfa06.txt"};
//   char nfa_file[][15] = {"bitnfa00.txt", "bitnfa01.txt", "bitnfa02.txt", "bitnfa03.txt", "bitnfa04.txt", "bitnfa05.txt", "bitnfa06.txt"};
   int i = 0;

   
   if (argc != 2)
   {
      usage_and_die("Missing input file");
   }

   if ( (f=fopen(argv[1], "r")) == NULL)
   {
      printf("could not open file %s\n", argv[1]);
      exit(0);
   }

   p = fgets(buf, 4096, f);
   while (p != NULL)
   {
      p[strlen(p)-1] = '\0';  /* get rid of trailing \n */
      RE = parse_regex(buf, strlen(buf), line++);
      fprintf(stdout, "\n\n\n--->RE: %s\n", p);

      fprintf(stdout, "\nThe parsed tree:\n");
      fprintf(stdout, "----------------\n");
      RE->printtree(stdout, 0);

      fprintf(stdout, "\nReconstruction from tree:\n");
      fprintf(stdout, "-------------------------\n");
      fprintf(stdout, "/");
      RE->unparse(stdout, 0);
      fprintf(stdout, "/\n");      

      fprintf(stdout, "\nThe NFA:\n");
      fprintf(stdout, "--------\n");
      NFA = re_tree_to_nfa(RE, line-1);
      VEC_NFA.push_back(NFA);
      //NFA->print_machine(stdout);
      NFA->xfa_output(stdout, 42);
      //NFA->bdd_output(nfa_file[i], 42);
      //NFA->xfa_output("the_nfa.nfa", 42);
      NFA->xfa_output(nfa_file[i], 42);

      fprintf(stdout, "\nThe DFA:\n");
      fprintf(stdout, "--------\n");
      DFA = NFA->make_dfa();
      //VEC_DFA.push_back(DFA);
      //DFA->print_machine(stdout);
      //DFA->xfa_output(stdout, 42);
      DFA->bdd_output(dfa_file[i], 42);
      i++;
      //DFA->xfa_output("the_dfa.dfa", 42);

      //fprintf(stdout, "\nThe Minimized DFA:\n");
      //fprintf(stdout, "------------------\n");
      //nfa_minimize(DFA);
      //DFA->bdd_output(dfa_file[i], 42);
      //VEC_DFA.push_back(DFA);
      //DFA->print_machine(stdout);
      //DFA->xfa_output(stdout, 42);
      //DFA->xfa_output("the_dfa_min.dfa", 42);

      delete RE;
//      delete NFA;
//      delete DFA;

      p = fgets(buf, 4096, f);
   }

   fprintf(stdout, "\n\n\nMinimization Test\n");
   fprintf(stdout, "-----------------\n");
   fprintf(stdout, "\nOriginal machine:\n");
   fprintf(stdout, "-----------------\n");
   nfa_t *m = dfa_min1();
   m->xfa_output(stdout, 43);
   //m->print_machine(stdout);

   fprintf(stdout, "\nMinimized machine:\n");
   fprintf(stdout, "------------------\n");
   nfa_minimize(m);
   //m->print_machine(stdout);
   m->xfa_output(stdout, 43);
   
   /* test NFA combination | (OR) */
   tmp_nfa = VEC_NFA[0];
   tmp_dfa = VEC_DFA[0];
   for (int i = 0; i < VEC_NFA.size() - 1; i++) {
     com_nfa = do_alternation_acc(tmp_nfa, VEC_NFA[i+1]);
     tmp_nfa = com_nfa;
     com_dfa2 = combine(tmp_dfa, VEC_DFA[i+1]);
     tmp_dfa = com_dfa2;
   }
   com_nfa->bdd_output("combined_nfa_bdd.txt", 42);
   com_nfa->xfa_output("combined_nfa_xfa.txt", 42);
   com_dfa = com_nfa->make_dfa();
   com_dfa->bdd_output("combined_dfa_bdd.txt", 42);
   nfa_minimize(com_dfa2);	/* used for signature match */
   com_dfa2->bdd_output("combined_dfa_bdd2.txt", 42);
   com_dfa2->xfa_output("combined_dfa_xfa.txt", 42);
   cout << "num_accept of combined nfa " << com_nfa->accepting.size() << endl;
   std::list<state_id_t>::iterator it = com_nfa->accepting.begin();
   while (it != com_nfa->accepting.end()) {
     fprintf(stdout, "%u ", *it);
     it++;
   }
   fprintf(stdout, "\n");

   fclose(f);

   return 1;
}


/*-----------------------------------------------------------------------------
 * usage_and_die
 *---------------------------------------------------------------------------*/
void usage_and_die(const char *msg)
{
   fprintf(stderr, "error: %s\n", msg);
   fprintf(stderr, "\nusage: main <inputfile>\n");

   fprintf(stderr, "\n"
   "This program is a simple front-end that tests the regex parser.  The\n"
   "input file contains a list of regex's, one per line, and the program\n"
   "parses each of these in turn and unparses the resulting tree.  The\n"
   "unparsed output is sent to stdin.\n");

   fprintf(stderr, "\n"
   "An example input file is the following:\n"
   "\n"
   "       /[\\d./*]{5,}\\wx+-y|^(\\d:\\d:\\d\\/)/   \n"
   "       /([^\\n]*?<){55}/  \n");

   fprintf(stderr, "\n\n\n");
   exit(0);
}


nfa_t *dfa_min1()
{
   nfa_t *n = new nfa_t();
   state_id_t s[8];
   unsigned int i;

   for (i=0; i < 8; i++)
   {
      s[i] = n->add_state();
   }

   n->start = s[0];

   for (i=0; i < 128; i++)
   {
      n->add_trans(s[0], s[1], i);
      n->add_trans(s[1], s[6], i);
      n->add_trans(s[2], s[0], i);
      n->add_trans(s[3], s[2], i);
      n->add_trans(s[4], s[7], i);
      n->add_trans(s[5], s[2], i);
      n->add_trans(s[6], s[6], i);
      n->add_trans(s[7], s[6], i);
   }

   for (i=128; i < 256; i++)
   {
      n->add_trans(s[0], s[5], i);
      n->add_trans(s[1], s[2], i);
      n->add_trans(s[2], s[2], i);
      n->add_trans(s[3], s[6], i);
      n->add_trans(s[4], s[5], i);
      n->add_trans(s[5], s[6], i);
      n->add_trans(s[6], s[4], i);
      n->add_trans(s[7], s[2], i);
   }

   n->accepting.push_back(2);
   n->states[s[2]].accept_id.push_back(42);

   if (!n->is_deterministic())
   {
      fprintf(stderr, "ERROR:  dfa_min1 construction not deterministic!\n");
      exit(0);
   }

   return n;
}
