/*  This program reads from the intermediate trace files generated by the monitor.
	Creates trace files that can be fed to Daikon directly

	Trace is generated according to the following granularity
	=> Class level
	=> Object level (identified by addr/name)
	=> Sequence level (arrays, linked lists)
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <string>
#include <sstream>
#include <map>
#include <dirent.h>
#include <vector>
using namespace std;

#define FILE_PATH "../snapshots"
#define CLASS_GRANULARITY "class"
#define OBJECT_GRANULARITY "object"
#define SEQUENCE_GRANULARITY "sequence"
#define MAX_BUFFER_SIZE 50000
#define ADDR_OR_NAME 1  // 0 - Use addr, 1- Use Name , for program points in trace file
#define LEFT_FIELD_DELIMITER '@'
#define RIGHT_FIELD_DELIMITER '|'
#define PPT ":::"
#define PARTITION_SIZE_FOR_LISTS 15
#define PARTITION_SIZE_FOR_DECLS  40  //60
#define PARTITION_SIZE_FOR_TRACES  30  //45

typedef struct 
{
	string name;
	string array;
} map_type;

void create_daikon_decls(string file_prefix);
void create_daikon_traces(string file_prefix);
void convert_trace_file(string file_prefix, string orig_prefix);
char * get_field_info(char * buffer, short field_num);
void create_daikon_traces_and_decls();
map<string,short> ppts;

string for_files, granularity;
int num_traces;

int main(int argc, char ** argv)
{
	if (argc != 4)
	{
        printf("USAGE: genDaikonTrace <name of ds | all> <granularity [class|object|sequence]> <num_traces>}\n");
        exit(-1);
    }

	for_files = argv[1];
	granularity = argv[2];	
	num_traces = atoi(argv[3]);

	cout << for_files << " " << granularity << " " << num_traces << endl;

	if(for_files == "all")
	{
		DIR * dip;
		dirent * dentry;

		dip = opendir(FILE_PATH);	
		if(dip == NULL)
		{
			cout << "Cannot open directory " << FILE_PATH << endl;
			exit(-1);
		}
		
		while ((dentry = readdir(dip)) != NULL)
        {
			//Locate the intermediate declaration file
            if(strstr(dentry->d_name,".ir.decls") != NULL)
			{
				//Get prefix
				char dit_name[255];
				strcpy(dit_name, dentry->d_name);

				//cout << dentry->d_name << endl;
				string file_prefix = strtok(dit_name, ".");

				if(granularity == CLASS_GRANULARITY)
				{
					//Convert .ir.decls file to .decls Daikon declaration file 
					create_daikon_decls(file_prefix);
					create_daikon_traces(file_prefix);
				}
				else if(granularity == OBJECT_GRANULARITY)
				{
					//First generate trace files, then decls file - optimized
					create_daikon_traces(file_prefix);
					create_daikon_decls(file_prefix);
				}
			}

        }	
	}
	else
	{
		string declfile = for_files;
		//For a single file
		if(granularity == CLASS_GRANULARITY)
		{
			create_daikon_decls(declfile);
			create_daikon_traces(for_files);
		}
		else if(granularity == OBJECT_GRANULARITY)
		{
			//First generate trace files, then decls file - optimized
			create_daikon_traces(for_files);
			create_daikon_decls(declfile);
		}
	}
	if(granularity == SEQUENCE_GRANULARITY)
	{
		create_daikon_traces_and_decls();
	}
	
	return 0;
}

void create_daikon_decls(string file_prefix)
{
	FILE * fd_in;
	FILE * fd_out;

	string filein = FILE_PATH ;
	filein = filein + "/" + file_prefix + ".ir.decls";

	fd_in = fopen(filein.c_str(),"r");
	if(fd_in == NULL)
	{
		cout << "Cannot open file " << filein << endl;
		exit(-1);
	}

	string fileout = FILE_PATH;
	fileout = fileout + "/" + file_prefix + ".decls";
	fd_out = fopen(fileout.c_str(),"w");
	if(fd_out == NULL)
	{
		cout << "Cannot open file " << fileout << endl;
		exit(-1);
	}

	//Read from .ir.decls file, write to .decls file
	if(granularity == CLASS_GRANULARITY)
	{
		//Only one program point is required
		int line_count=0;
		char buffer[MAX_BUFFER_SIZE];
		fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
		int decl_lines = 0;
		char classname[MAX_BUFFER_SIZE];
		int part_number=0;

		while(!feof(fd_in))
		{
			line_count++;
			if(line_count == 2)
			{
				buffer[strlen(buffer) - 1] ='\0';
				strcpy(classname,buffer);
				strcat(buffer,":::classdecl\n");
				decl_lines = 0;
			}
			if(decl_lines >= PARTITION_SIZE_FOR_DECLS)
			{
				fputs(buffer,fd_out);
				part_number++;
				stringstream tmp;
				tmp << part_number;
				strcpy(buffer,"\nDECLARE\n");
				strcat(buffer,classname);
				strcat(buffer,(tmp.str()).c_str());
				strcat(buffer,":::classdecl\n");
				decl_lines = 0;
				
			}
			fputs(buffer,fd_out);
			fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
			decl_lines++;

		}
	}
	else if(granularity == OBJECT_GRANULARITY)
	{
		//Expects that the Map ppts is populated with program point names
		//Read each declaration and print them as many times as there are program points
		map<string,short>::iterator iter = ppts.begin();
		cout << "ppts map size = "<< ppts.size() << endl;
		while(iter != ppts.end())
		{
			int line_count =0;
			int part_number =0;
			char buffer[MAX_BUFFER_SIZE];
			int decl_lines = 0;
			fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
			while(!feof(fd_in))
        	{
				line_count++;
				if(line_count == 1)
				{
					fputs("\n",fd_out);
				}
				if((line_count == 2) || (decl_lines >= PARTITION_SIZE_FOR_DECLS))
				{
					if(decl_lines >= PARTITION_SIZE_FOR_DECLS)
						fputs(buffer,fd_out);

					stringstream tmp;
					char tmpbuf[MAX_BUFFER_SIZE];
					tmp << part_number;

					strcpy(tmpbuf,(iter->first).c_str());
					string classname = strtok(tmpbuf,":");
					string addrname = strtok(NULL,":");

					if(decl_lines >= PARTITION_SIZE_FOR_DECLS)
						strcpy(buffer,"\nDECLARE\n");
					else
					{
						strcpy(buffer,"");							
					}

					strcat(buffer,classname.c_str());
					strcat(buffer,tmp.str().c_str());
					strcat(buffer,":::");
					strcat(buffer,addrname.c_str());

					if(ADDR_OR_NAME == 0)
							strcat(buffer,"\n");

					part_number++;
					decl_lines = 0;
				}

				decl_lines++;
				fputs(buffer,fd_out);
				fgets(buffer,MAX_BUFFER_SIZE,fd_in);
            }

			++iter;
			fseek(fd_in, 0,SEEK_SET);			
		}
	}
	ppts.clear();
	fclose(fd_in);
	fclose(fd_out);
}


void create_daikon_traces(string file_prefix)
{
	//Read from .ir.dtrace file, write to .dtrace file
	for(int i=1; i<= num_traces; i++)
	{
		
		string trcfile =  file_prefix ;
		std::stringstream strnum;
		strnum << i;

		trcfile.append(strnum.str());
		convert_trace_file(trcfile, file_prefix);
	}
}

bool omit(string type)
{
	vector<string> black_list;

	/*black_list.push_back("struct __wait_queue");
	black_list.push_back("struct block_device");
	black_list.push_back("struct char_device");
	black_list.push_back("struct net_device");
	black_list.push_back("struct free_area_struct");
	black_list.push_back("struct page"); */

	for(int i=0; i< black_list.size(); i++)
	{
		if(black_list[i] == type)
		{
			return true;
		}
	}
	return false;
}

void create_daikon_traces_and_decls()
{
	string file_prefix ="linkedlist";
	multimap<string,map_type> lists;

    //Read from .ir.dtrace file, write to .dtrace file
    for(int i=1; i<= num_traces; i++)
	{
        string trcfile; 
        std::stringstream strnum;
        strnum << FILE_PATH << "/" << file_prefix << i;
        trcfile.append(strnum.str());
		string commonprefix = trcfile;
		trcfile = trcfile + ".ir.dtrace";

		FILE * fd_in = fopen(trcfile.c_str(),"r");
		if(fd_in == NULL)
		{
			cout << "Cannot open file " << trcfile.c_str() << endl;
			return;
		}

		char buffer[MAX_BUFFER_SIZE];
		fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
		while(!feof(fd_in))
		{
			if(strstr(buffer,PPT) != NULL)
			{
				string name = strtok(buffer,PPT);
				string type = strtok(NULL,PPT);	

				fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
				
				map_type m;
				m.name = name;
				m.array = buffer;
				lists.insert(make_pair(type,m));	
			}	
			fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
		}	

		//Create declaration and trace in one file
		string declfile = commonprefix + ".dtrace";
		FILE * fd_out = fopen(declfile.c_str(),"w");
		if(fd_out == NULL)
		{
            cout << "Cannot open file " << declfile.c_str() << endl;
            return;
        }
		
		multimap<string,map_type>::iterator iter = lists.begin();

		//Generate declarations here
		while(iter != lists.end())
		{
			pair<multimap<string,map_type>::iterator, multimap<string,map_type>::iterator> p;
			multimap<string,map_type>::iterator subiter;
			int part_number =0;
			
			if(!omit(iter->first))
			{
			cout << " ppname = [" << iter->first << "]" << endl;

			fprintf(fd_out, "\n\nDECLARE\n");
			fprintf(fd_out, "list%d:::%s", part_number, (iter->first).c_str()); //Program point name
			++part_number;

			p = lists.equal_range(iter->first);
			subiter = p.first;
			int count_lists =0;
			while(subiter != p.second)
			{
				map_type m = subiter->second;	
				fprintf(fd_out, "%s[]\n", m.name.c_str());
				fprintf(fd_out,"hashcode[]\nhashcode[]\n0\n");
				++subiter;
				++count_lists;

				if((count_lists == PARTITION_SIZE_FOR_LISTS) && (subiter != p.second))
				{
					//Declare another program point

					fprintf(fd_out, "\n\nDECLARE\n");
					fprintf(fd_out, "list%d:::%s", part_number, (iter->first).c_str()); //Program point name
					++part_number;

					count_lists=0;
				}
			}
			}
			iter=lists.upper_bound(iter->first);
		}

		fprintf(fd_out,"\n\n");
		cout << "Finished writing declarations " << endl;

		//Generate traces here
		iter = lists.begin();

		while(iter != lists.end())
        {
            pair<multimap<string,map_type>::iterator, multimap<string,map_type>::iterator> p;
            multimap<string,map_type>::iterator subiter;
			int part_number =0;
                                                                                                                                               
			if(!omit(iter->first))
			{
			//List the program point name
            fprintf(fd_out, "\nlist%d:::%s", part_number, (iter->first).c_str());
			++part_number;
                                                                                                                                               
            p = lists.equal_range(iter->first);
            subiter = p.first;
			int count_lists =0;
            while(subiter != p.second)
            {
                map_type m = subiter->second;
                fprintf(fd_out, "%s[]\n", m.name.c_str());
                fprintf(fd_out, "%s", m.array.c_str());
                fprintf(fd_out,"1\n");
                ++subiter;
				++count_lists;

				if((count_lists == PARTITION_SIZE_FOR_LISTS) && (subiter != p.second))
				{
					//Make it part of the next program point
					//List the program point name
            		fprintf(fd_out, "\nlist%d:::%s", part_number, (iter->first).c_str());
					++part_number;

					count_lists=0;
				}
            }
			}
			iter=lists.upper_bound(iter->first);
        }

		cout << "Finished writing traces" << endl;

		lists.clear();
    }
}



//fileprefix here is file_prefix + num , orig_prefix is without num
void convert_trace_file(string file_prefix, string orig_prefix)
{
	FILE * fd_in;
	FILE * fd_out;

	string filein = FILE_PATH ;
	filein = filein + "/" + file_prefix + ".ir.dtrace";

	fd_in = fopen(filein.c_str(),"r");
	if(fd_in == NULL)
	{
		cout << "Cannot open file " << filein << endl;
		return;
	}

	string fileout = FILE_PATH;
	fileout = fileout + "/" + file_prefix + ".dtrace";
	fd_out = fopen(fileout.c_str(),"w");
	if(fd_out == NULL)
	{
		cout << "Cannot open file " << fileout << endl;
		exit(-1);
	}

	char buffer[MAX_BUFFER_SIZE];
	fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
	int decl_lines =0;
	int part_number=0;
	string orig_pp_name, pp_name, addr, name; 
	while(!feof(fd_in))
	{
		//string srch = LEFT_FIELD_DELIMITER + orig_prefix + RIGHT_FIELD_DELIMITER;
		string srch = orig_prefix + ":";

		if(strstr(buffer,srch.c_str()) != NULL)
		{
			if(granularity == CLASS_GRANULARITY)
			{
				strcpy(buffer,"\n");
				strcat(buffer,orig_prefix.c_str());
				strcat(buffer,":::classdecl");
				strcat(buffer,"\n");
				part_number =0;
            	decl_lines = 0;
			}
			else if(granularity == OBJECT_GRANULARITY)
			{
				part_number =0;
            	stringstream tmp;
            	tmp << part_number;
				//string pp_name, addr, name; 
				orig_pp_name = strtok(buffer,":");
				addr= strtok(NULL,":");
				name = strtok(NULL,":");

				string pp_ins_name;

				if(ADDR_OR_NAME == 0)
				{ //Use addr
						pp_name = orig_pp_name + tmp.str() + ":::" + addr;
						pp_ins_name = orig_pp_name + ":::" + addr;
				}
				else
				{ //Use name
						pp_name = orig_pp_name + tmp.str() + ":::" + name;
						pp_ins_name = orig_pp_name + ":::" + name;
				}

				//ppts.insert(make_pair(pp_name,1));
				ppts.insert(make_pair(pp_ins_name,1));
				strcpy(buffer,"\n");
				strcat(buffer, pp_name.c_str());
				if(ADDR_OR_NAME == 0)
					strcat(buffer,"\n");
            	decl_lines = 0;
			}
		}
		
		if(decl_lines >= PARTITION_SIZE_FOR_TRACES)
        {
			fputs(buffer,fd_out);
            part_number++;
            stringstream tmp;
            tmp << part_number;
            strcpy(buffer,"\n");
			if(granularity == CLASS_GRANULARITY)
			{
            	strcat(buffer,orig_prefix.c_str());
            	strcat(buffer,(tmp.str()).c_str());
            	strcat(buffer,":::classdecl\n");
			}
			else if(granularity == OBJECT_GRANULARITY)
			{
				string new_pp_name;
				if(ADDR_OR_NAME == 0)
				{ //Use addr
						new_pp_name = orig_pp_name + tmp.str() + ":::" + addr;
				}
				else
				{ //Use name
						new_pp_name = orig_pp_name + tmp.str() + ":::" + name;
				}
				strcat(buffer, new_pp_name.c_str());
				if(ADDR_OR_NAME == 0)
					strcat(buffer,"\n");
			}
            decl_lines = 0;
        }

		fputs(buffer,fd_out);
		decl_lines++;
		fgets(buffer,MAX_BUFFER_SIZE,fd_in);	
	}
	fclose(fd_in);
	fclose(fd_out);
}

char * get_field_info(char * buffer, short field_num)
{
    char l_delimiter =LEFT_FIELD_DELIMITER;
    char r_delimiter =RIGHT_FIELD_DELIMITER;
    short field_counter =0;
    char * ret_buf;
    short start_copying = 0, copied =0;
    int j=0;
    int i;
                                                                                                                                               
	cout << buffer << endl;
    ret_buf = (char *)malloc(MAX_BUFFER_SIZE);
    for(i=0; i< strlen(buffer); i++)
    {
        if(buffer[i] == l_delimiter)
        {
            field_counter++;
            if(field_counter == field_num) //This is the field to return
            {
                start_copying = 1;
            }
        }
        else if(buffer[i] == r_delimiter)
        {
            if(field_counter == field_num)
            {
                assert(start_copying == 1);
                *(ret_buf + j) = '\0';
                copied = 1;
                break;
            }
        }
        else
        {
            if(start_copying == 1)
            {
                *(char *)(ret_buf + j) = buffer[i];
                                                                                                                                               
                j++;
            }
        }
    }
    if(copied == 1)
        return(ret_buf);
    else
        return(NULL);
}
