#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>

extern "C" {
#include <xenctrl.h>
#include "xg_save_restore.h"
#include "xg_private.h"
}

/* Number of xen_pfn_t in a page */
//#define FPP             (PAGE_SIZE/(guest_width))

/* Number of entries in the pfn_to_mfn_frame_list_list */
//#define P2M_FLL_ENTRIES (((p2m_size)+(FPP*FPP)-1)/(FPP*FPP))

/* number of pfns this guest has (i.e. number of entries in the P2M) */
static unsigned long p2m_size;

/* Live mapping of the table mapping each PFN to its current MFN. */
static xen_pfn_t *live_p2m = NULL;
static xen_pfn_t *my_p2m = NULL;

/* Live mapping of system MFN to PFN table. */
static xen_pfn_t *live_m2p = NULL;
static unsigned long m2p_mfn0;

/* Double and single indirect references to the live P2M table */
void *live_p2m_frame_list_list = NULL;
void *live_p2m_frame_list = NULL;

/* Copies of the above. */
xen_pfn_t *p2m_frame_list_list = NULL;
xen_pfn_t *p2m_frame_list = NULL;

/*
** Map the top-level page of MFNs from the guest. The guest might not have
** finished resuming from a previous restore operation, so we wait a while for
** it to update the MFN to a reasonable value.
*/
static void *map_frame_list_list(int xc_handle, uint32_t dom, unsigned int guest_width,
                                 shared_info_any_t *shinfo)
{
    int count = 100;
    void *p;
    uint64_t fll = GET_FIELD(shinfo, arch.pfn_to_mfn_frame_list_list);

    while ( count-- && (fll == 0) )
    {
        usleep(10000);
        fll = GET_FIELD(shinfo, arch.pfn_to_mfn_frame_list_list);
    }

    if ( fll == 0 )
    {
        ERROR("Timed out waiting for frame list updated.");
        return NULL;
    }

    p = xc_map_foreign_range(xc_handle, dom, PAGE_SIZE, PROT_READ, fll);
    if ( p == NULL )
        ERROR("Couldn't map p2m_frame_list_list (errno %d)", errno);

    return p;
}

static xen_pfn_t *map_p2m_table(int xc_handle, uint32_t dom, unsigned long p2m_size, unsigned int guest_width, shared_info_any_t *live_shinfo)
{
	vcpu_guest_context_any_t ctxt;
	
	/* The mapping of the live p2m table itself */
	xen_pfn_t *p2m = NULL;
	
	int i, success = 0;

	live_p2m_frame_list_list = map_frame_list_list(xc_handle, dom, guest_width,
                                                   live_shinfo);
	if ( !live_p2m_frame_list_list )
		goto out;
	
	/* Get a local copy of the live_P2M_frame_list_list */
	if ( !(p2m_frame_list_list = (xen_pfn_t*)malloc(PAGE_SIZE)) )
	{
		ERROR("Couldn't allocate p2m_frame_list_list array");
		goto out;
	}
	memcpy(p2m_frame_list_list, live_p2m_frame_list_list, PAGE_SIZE);
	
	/* Canonicalize guest's unsigned long vs ours */
	if ( guest_width > sizeof(unsigned long) )
		for ( i = 0; i < PAGE_SIZE/sizeof(unsigned long); i++ )
			if ( i < PAGE_SIZE/guest_width )
				p2m_frame_list_list[i] = ((uint64_t *)p2m_frame_list_list)[i];
			else
				p2m_frame_list_list[i] = 0;
	else if ( guest_width < sizeof(unsigned long) )
		for ( i = PAGE_SIZE/sizeof(unsigned long) - 1; i >= 0; i-- )
			p2m_frame_list_list[i] = ((uint32_t *)p2m_frame_list_list)[i];
	
	live_p2m_frame_list =
		xc_map_foreign_batch(xc_handle, dom, PROT_READ,
						p2m_frame_list_list,
						P2M_FLL_ENTRIES);
	if ( !live_p2m_frame_list )
	{
		ERROR("Couldn't map p2m_frame_list");
		goto out;
	}
	
	/* Get a local copy of the live_P2M_frame_list */
	if ( !(p2m_frame_list = (xen_pfn_t*)malloc(P2M_TOOLS_FL_SIZE)) )
	{
		ERROR("Couldn't allocate p2m_frame_list array");
		goto out;
	}
	memset(p2m_frame_list, 0, P2M_TOOLS_FL_SIZE);
	memcpy(p2m_frame_list, live_p2m_frame_list, P2M_GUEST_FL_SIZE);
	
	/* Canonicalize guest's unsigned long vs ours */
	if ( guest_width > sizeof(unsigned long) )
		for ( i = 0; i < P2M_FL_ENTRIES; i++ )
			p2m_frame_list[i] = ((uint64_t *)p2m_frame_list)[i];
	else if ( guest_width < sizeof(unsigned long) )
		for ( i = P2M_FL_ENTRIES - 1; i >= 0; i-- )
			p2m_frame_list[i] = ((uint32_t *)p2m_frame_list)[i];
	
	
	/* Map all the frames of the pfn->mfn table. For migrate to succeed,
		the guest must not change which frames are used for this purpose.
		(its not clear why it would want to change them, and we'll be OK
		from a safety POV anyhow. */
	
	p2m = (xen_pfn_t*)xc_map_foreign_batch(xc_handle, dom, PROT_READ,
							p2m_frame_list,
							P2M_FL_ENTRIES);
	if ( !p2m )
	{
		ERROR("Couldn't map p2m table");
		goto out;
	}
	live_p2m = p2m; /* So that translation macros will work */


	success = 1;

out:

	if ( !success && p2m )
		munmap(p2m, P2M_FLL_ENTRIES * PAGE_SIZE);

	if ( live_p2m_frame_list_list )
		munmap(live_p2m_frame_list_list, PAGE_SIZE);
	
	if ( live_p2m_frame_list )
		munmap(live_p2m_frame_list, P2M_FLL_ENTRIES * PAGE_SIZE);
	
	if ( p2m_frame_list_list ) 
		free(p2m_frame_list_list);
	
	if ( p2m_frame_list ) 
		free(p2m_frame_list);

	if ( !success && p2m )
		munmap(p2m, P2M_FLL_ENTRIES * PAGE_SIZE);
	
	return success ? p2m : NULL;
}

xen_pfn_t *xc_map_m2p(int xc_handle,
                                 unsigned long max_mfn,
                                 int prot,
                                 unsigned long *mfn0)
{
    struct xen_machphys_mfn_list xmml;
    privcmd_mmap_entry_t *entries;
    unsigned long m2p_chunks, m2p_size;
    xen_pfn_t *m2p;
    xen_pfn_t *extent_start;
    int i;

    m2p = NULL;
    m2p_size   = M2P_SIZE(max_mfn);
    m2p_chunks = M2P_CHUNKS(max_mfn);

    //printf("m2p_size: 0x%x m2p_chunks: 0x%x max_mfn: 0x%x\n",m2p_size,m2p_chunks,max_mfn);

    xmml.max_extents = m2p_chunks;

    extent_start = (xen_pfn_t *)calloc(m2p_chunks, sizeof(xen_pfn_t));
    if ( !extent_start )
    {
        ERROR("failed to allocate space for m2p mfns");
        goto err0;
    }
    set_xen_guest_handle(xmml.extent_start, extent_start);

    if ( xc_memory_op(xc_handle, XENMEM_machphys_mfn_list, &xmml) ||
         (xmml.nr_extents != m2p_chunks) )
    {
        ERROR("xc_get_m2p_mfns");
        goto err1;
    }

    entries = (privcmd_mmap_entry_t *)calloc(m2p_chunks, sizeof(privcmd_mmap_entry_t));
    if (entries == NULL)
    {
        ERROR("failed to allocate space for mmap entries");
        goto err1;
    }

    for ( i = 0; i < m2p_chunks; i++ )
        entries[i].mfn = extent_start[i];

    m2p = (xen_pfn_t *)xc_map_foreign_ranges(xc_handle, DOMID_XEN,
			m2p_size, prot, M2P_CHUNK_SIZE,
			entries, m2p_chunks);
    if (m2p == NULL)
    {
        ERROR("xc_mmap_foreign_ranges failed");
        goto err2;
    }

    if (mfn0)
        *mfn0 = entries[0].mfn;

err2:
    free(entries);
err1:
    free(extent_start);

err0:
    return m2p;
}

xen_pfn_t * get_p2m_table(int xc_handle, uint32_t dom)
{
	xc_dominfo_t info;
	xen_pfn_t *p2m;
	/* Live mapping of shared info structure */
	shared_info_any_t *live_shinfo = NULL;
	/* The new domain's shared-info frame number. */
	unsigned long shared_info_frame;
	/* max mfn of the whole machine */
	static unsigned long max_mfn;
	
	/* virtual starting address of the hypervisor */
	static unsigned long hvirt_start;
	
	/* #levels of page tables used by the current guest */
	static unsigned int pt_levels;
	
	/* Address size of the guest */
	unsigned int guest_width;

	if ( !get_platform_info(xc_handle, dom,
					&max_mfn, &hvirt_start, &pt_levels, &guest_width) )
	{
		fprintf(stderr,"Unable to get platform info.");
		exit(-1);
	}

	if ( xc_domain_getinfo(xc_handle, dom, 1, &info) != 1 )
	{
		fprintf(stderr,"Could not get domain info");
		exit(-1);
	}

	shared_info_frame = info.shared_info_frame;

	/* Map the shared info frame */
	live_shinfo = (shared_info_any_t *)xc_map_foreign_range(xc_handle, dom, PAGE_SIZE,
								PROT_READ, shared_info_frame);
	if ( !live_shinfo )
	{
		fprintf(stderr,"Couldn't map live_shinfo");
		exit(-1);
	}

	/* Get the size of the P2M table */
	p2m_size = xc_memory_op(xc_handle, XENMEM_maximum_gpfn, &dom) + 1;
	printf("p2m_size: 0x%x\n",p2m_size);

	/* Map the P2M table, and write the list of P2M frames */
	p2m = map_p2m_table(xc_handle, dom, p2m_size, guest_width, live_shinfo);
	if ( p2m == NULL )
	{
		fprintf(stderr,"Failed to map the p2m frame list\n");
		exit(-1);
	}

	return p2m;
}

xen_pfn_t * get_m2p_table(int xc_handle, uint32_t dom)
{
	xen_pfn_t * m2p;
	
	/* max mfn of the whole machine */
	static unsigned long max_mfn;
	/* virtual starting address of the hypervisor */
	static unsigned long hvirt_start;
	/* #levels of page tables used by the current guest */
	static unsigned int pt_levels;
	/* Address size of the guest */
	unsigned int guest_width;

	if ( !get_platform_info(xc_handle, dom,
					&max_mfn, &hvirt_start, &pt_levels, &guest_width) )
	{
		fprintf(stderr,"Unable to get platform info.");
		exit(-1);
	}

	if ( !(m2p = xc_map_m2p(xc_handle, max_mfn, PROT_READ, &m2p_mfn0)) )
	{
		ERROR("Failed to map live M2P table");
		exit(-1);
	}

	return m2p;
}

//unsigned char * build_memory_map(int domain, xen_pfn_t *monitor_p2m, xen_pfn_t *monitor_m2p)
unsigned char * build_memory_map(uint32_t domain)
{
	int xc_handle;
	xc_dominfo_t info;
	uint32_t dom;
	int i;
	unsigned char *memory;
	dom = domain;
	
	//printf("domain: %d\n",dom);

	xc_handle = xc_interface_open();
	
	if(xc_handle == -1)
	{
		fprintf(stderr, "Unable to get handle to hypervisor\n");
		exit(-1);
	}

	live_p2m = get_p2m_table(xc_handle, dom);

	live_m2p = get_m2p_table(xc_handle, dom);

	/* Copy live_p2m to my_p2m, pass my_p2m to mmap*/

	my_p2m = (xen_pfn_t *)malloc(p2m_size * sizeof(xen_pfn_t));
	memcpy(my_p2m,live_p2m,p2m_size * sizeof(xen_pfn_t));

	//printf("live_p2m: 0x%x live_m2p: 0x%x\n",live_p2m[0],live_m2p[0]);

 	memory = (unsigned char *)xc_map_foreign_batch(xc_handle, dom, PROT_READ,
                           my_p2m, p2m_size );

	if(memory == NULL)
	{
		fprintf(stderr, "Failed to map memory\n");
		exit(-1);
	}

	//monitor_p2m = &(*live_p2m);
	//monitor_m2p = &(*live_m2p);
	/* unmap p2m table */
	/*if ( live_p2m )
		munmap(live_p2m, P2M_FLL_ENTRIES * PAGE_SIZE);*/

	xc_interface_close(xc_handle);

	return memory;
}

